"""This suite of tests is intended to test the correctness of representation
implementations within the graphtypes module.
"""
import unittest

from testing.testutils import StreamMocker

from graphadt.representations import AdjacencyLists, AdjacencyMatrix


def suite():
    return unittest.TestLoader().loadTestsFromName(__name__)


class EmptyTests(object):
    """This class contains the common code for testing empty and invalid conditions in a representation.
    Requires that self.newRepresentation() is a constructor of a concrete instance of Graph
    """

    def setUp(self):
        self.__r = self.newRepresentation()  # Default constructor, leaves graph empty

    def testOrder0(self):
        assert self.__r.order() == 0, repr(self.__r) + " has vertices"

    def testSize0(self):
        assert self.__r.size() == 0, repr(self.__r) + " has edges"

    def testSelfEdgesSize0(self):
        assert self.__r.selfEdges() == 0, "Representation has self edges"

    def testAdd0Vertices(self):
        ord = self.__r.order()
        self.__r.addVertices(0)
        assert ord == self.__r.order(), repr(self.__r) + " has gained or lost vertices"

    def testAddArc(self):
        self.assertRaises(ValueError, self.__r.addArc, 0, 0)

    def testRemoveArc(self):
        self.assertRaises(ValueError, self.__r.removeArc, 0, 0)

    def testIsArc(self):
        self.assertRaises(ValueError, self.__r.isArc, 0, 0)

    def testDegree(self):
        self.assertRaises(ValueError, self.__r.degree, 0)

    def testInDegree(self):
        self.assertRaises(ValueError, self.__r.inDegree, 0)

    def testNeighbours(self):
        self.assertRaises(ValueError, self.__r.neighbours, 0)

    def testWritingEmpty(self):
        assert (
            str(self.__r)
            == """\
0
"""
        )


class StandardTests(object):
    def setUp(self):
        self.__r = self.newRepresentation()  # Initially empty representation

        self.__s = self.newRepresentation()  # 1 vertex
        self.__s.addVertices(1)

        self.__t = self.newRepresentation()  # 2 Vertices, one arc
        self.__t.addVertices(2)
        self.__t.addArc(0, 1)

        self.__u = self.newRepresentation()  # 5 Vertices, complex arcs and edges
        self.__u.addVertices(5)
        for x in range(5):
            for y in range(5):
                if not y or not x % y:  # Zero and 1 connect to everything
                    self.__u.addArc(y, x)  # Arc between i and j implies i divides j

    def testOrderAdd1VertexEmptyRepresentation(self):
        assert self.__r.order() == 0, "Representation is not correct order"
        self.__r.addVertices(1)
        assert self.__r.order() == 1, "Representation order is not 1"

    def testSizeAdd1VertexEmptyRepresentation(self):
        assert self.__r.size() == 0, "Representation is not correct size"
        self.__r.addVertices(1)
        assert self.__r.size() == 0, "Representation size is not 0"

    def testSelfEdgesAdd1VertexEmptyRepresentation(self):
        assert self.__s.selfEdges() == 0, "Representation is not correct size"
        self.__s.addVertices(1)
        assert self.__s.selfEdges() == 0, "Representation self edges is not 0"

    def testDegreeAdd1VertexEmptyRepresentation(self):
        self.__r.addVertices(1)
        assert self.__r.degree(0) == 0, "Degree of 0 is not 0"

    def testInDegreeAdd1VertexEmptyRepresentation(self):
        self.__r.addVertices(1)
        assert self.__r.inDegree(0) == 0, "InDegree of 0 is not 0"

    def testNeighboursAdd1VertexEmptyRepresentation(self):
        self.__r.addVertices(1)
        assert not self.__r.neighbours(0), "Vertex 0 has neighbours"

    def testOrderAdd1Vertex1VertexRepresentation(self):
        assert self.__s.order() == 1, "Representation is not correct order"
        self.__s.addVertices(1)
        assert self.__s.order() == 2, "Representation order is not 2"

    def testSizeAdd1Vertex1VertexRepresentation(self):
        assert self.__s.size() == 0, "Representation is not correct size"
        self.__s.addVertices(1)
        assert self.__s.size() == 0, "Representation size is not 0"

    def testDegreeAdd1Vertex1VertexRepresentation(self):
        self.__s.addVertices(1)
        assert self.__s.degree(1) == 0, "Degree of 1 is not 0"

    def testInDegreeAdd1Vertex1VertexRepresentation(self):
        self.__s.addVertices(1)
        assert self.__s.inDegree(1) == 0, "InDegree of 1 is not 0"

    def testNeighboursAdd1Vertex1VertexRepresentation(self):
        self.__s.addVertices(1)
        assert not self.__s.neighbours(1), "Vertex 1 has neighbours"

    ####
    def testOrderAdd1Vertex2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        self.__t.addVertices(1)
        assert self.__t.order() == 3, "Representation order is not 3"

    def testSizeAdd1Vertex2VertexRepresentation(self):
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.addVertices(1)
        assert self.__t.size() == 1, "Representation size is not 1"

    def testSelfEdgesAdd1Vertex2VertexRepresentation(self):
        assert self.__s.selfEdges() == 0, "Representation should not have self edges"
        self.__s.addVertices(1)
        assert self.__s.selfEdges() == 0, "Representation self edges is not 0"

    def testDegreeAdd1Vertex2VertexRepresentation(self):
        self.__t.addVertices(1)
        assert self.__t.degree(0) == 1, "degree of 0 is not 1"
        assert self.__t.degree(1) == 0, "degree of 1 is not 0"
        assert self.__t.degree(2) == 0, "degree of 2 is not 0"

    def testInDegreeAdd1Vertex2VertexRepresentation(self):
        self.__t.addVertices(1)
        assert self.__t.inDegree(0) == 0, "degree of 0 is not 0"
        assert self.__t.inDegree(1) == 1, "degree of 1 is not 1"
        assert self.__t.inDegree(2) == 0, "degree of 2 is not 0"

    def testNeighboursAdd1Vertex2VertexRepresentation(self):
        self.__t.addVertices(1)
        assert self.__t.neighbours(0) == [1], "neighbours of 0 is not 1"
        assert self.__t.neighbours(1) == [], "1 wrongly has neighbours"
        assert self.__t.neighbours(2) == [], "2 wrongly has neighbours"

    def testOrderAdd2Vertex2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        self.__t.addVertices(2)
        assert self.__t.order() == 4, "Representation order is not 4"

    def testSizeAdd2Vertex2VertexRepresentation(self):
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.addVertices(2)
        assert self.__t.size() == 1, "Representation size is not 1"

    def testIsArcAdd2Vertex2VertexRepresentation(self):
        self.__t.addVertices(2)
        assert self.__t.isArc(0, 1) == 1, "Connection destroyed"
        for x in range(4):
            for y in range(4):
                if x != 0 and y != 1:
                    assert not self.__t.isArc(x, y), "Invalid connection created"

    def testDegreeAdd2Vertex2VertexRepresentation(self):
        self.__t.addVertices(2)
        assert self.__t.degree(0) == 1, "Connection destroyed"
        assert self.__t.degree(1) == 0, "Invalid connection created"
        assert self.__t.degree(2) == 0, "Invalid connection created"
        assert self.__t.degree(3) == 0, "Invalid connection created"

    def testInDegreeAdd2Vertex2VertexRepresentation(self):
        self.__t.addVertices(2)
        assert self.__t.inDegree(0) == 0, "Invalid connection created"
        assert self.__t.inDegree(1) == 1, "connection destroyed"
        assert self.__t.inDegree(2) == 0, "Invalid connection created"
        assert self.__t.inDegree(3) == 0, "Invalid connection created"

    def testNeighboursAdd2Vertex2VertexRepresentation(self):
        self.__t.addVertices(2)
        assert self.__t.neighbours(0) == [1], "connection destroyed"
        assert self.__t.neighbours(1) == [], "Invalid connection created"
        assert self.__t.neighbours(2) == [], "Invalid connection created"
        assert self.__t.neighbours(3) == [], "Invalid connection created"

    #
    def testOrderAdd1Arc1VertexRepresentation(self):
        assert self.__s.order() == 1, "Representation is not correct order"
        self.__s.addArc(0, 0)
        assert self.__s.order() == 1, "Representation order is not 1"

    def testSizeAdd1Arc1VertexRepresentation(self):
        assert self.__s.size() == 0, "Representation is not correct size"
        self.__s.addArc(0, 0)
        assert self.__s.size() == 1, "Representation size is not 1"

    def testSelfEdgesAdd1Arc1VertexRepresentation(self):
        assert self.__s.selfEdges() == 0, "Representation is not correct size"
        self.__s.addArc(0, 0)
        assert self.__s.selfEdges() == 1, "Representation selfedges should be 1"

    def testIsArcAdd1Arc1VertexRepresentation(self):
        assert not self.__s.isArc(0, 0), "Representation has incorrect arc"
        self.__s.addArc(0, 0)
        assert self.__s.isArc(0, 0), "No arc between 0 and 0"

    def testDegreeAdd1Arc1VertexRepresentation(self):
        assert self.__s.degree(0) == 0, "Degree of 0 is incorrect"
        self.__s.addArc(0, 0)
        assert self.__s.degree(0) == 1, "Degree of 0 not 1"

    def testInDegreeAdd1Arc1VertexRepresentation(self):
        assert self.__s.inDegree(0) == 0, "indegree of 0 is incorrect"
        self.__s.addArc(0, 0)
        assert self.__s.inDegree(0) == 1, "indegree of 0 not 1"

    def testNeighboursAdd1Arc1VertexRepresentation(self):
        assert not self.__s.neighbours(0), "0 should have no neighbours"
        self.__s.addArc(0, 0)
        assert 0 in self.__s.neighbours(0), "0 is not it's own neighbour"

    #
    def testOrderAddArcIsIdempotent(self):
        assert self.__s.order() == 1, "Representation is not correct order"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert self.__s.order() == 1, "Representation order is not 1"

    def testSizeAddArcIsIdempotent(self):
        assert self.__s.size() == 0, "Representation is not correct size"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert self.__s.size() == 1, "Representation size is not 1"

    def testSelfEdgesAdd1Vertex2VertexRepresentation(self):
        assert self.__s.selfEdges() == 0, "Representation should not have self edges"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert self.__s.selfEdges() == 1, "Representation self edges is not 1"

    def testIsArcSizeAddArcIsIdempotent(self):
        assert not self.__s.isArc(0, 0), "Representation has incorrect arc"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert self.__s.isArc(0, 0), "No arc between 0 and 0"

    def testDegreeAddArcIsIdempotent(self):
        assert self.__s.degree(0) == 0, "Degree of 0 is incorrect"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert self.__s.degree(0) == 1, "Degree of 0 not 1"

    def testInDegreeAddArcIsIdempotent(self):
        assert self.__s.inDegree(0) == 0, "indegree of 0 is incorrect"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert self.__s.inDegree(0) == 1, "indegree of 0 not 1"

    def testNeighboursAddArcIsIdempotent(self):
        assert not self.__s.neighbours(0), "0 should have no neighbours"
        self.__s.addArc(0, 0)
        self.__s.addArc(0, 0)
        assert 0 in self.__s.neighbours(0), "0 is not it's own neighbour"

    #
    def testOrderAdd2Arcs2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        self.__t.addArc(1, 0)
        assert self.__t.order() == 2, "Representation order is not 4"

    def testSizeAdd2Arcs2VertexRepresentation(self):
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.addArc(1, 0)
        assert self.__t.size() == 2, "Representation size is not 1"

    def testSelfEdgesAdd2Arcs2VertexRepresentation(self):
        assert self.__t.selfEdges() == 0, "Representation should not have self edges"
        self.__t.addArc(1, 0)
        assert self.__t.selfEdges() == 0, "Representation self edges is not 0"

    def testDegreeAdd2Arcs2VertexRepresentation(self):
        assert self.__t.degree(0) == 1, "Vertex 0 has wrong degree"
        assert self.__t.degree(1) == 0, "Vertex 1 has wrong degree"
        self.__t.addArc(1, 0)
        assert self.__t.degree(0) == 1, "Vertex 0 has wrong degree"
        assert self.__t.degree(1) == 1, "Vertex 1 has wrong degree"

    def testIsArcAdd2Arcs2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.addArc(1, 0)
        assert self.__t.order() == 2, "Representation order is not 4"
        assert self.__t.size() == 2, "Representation size is not 1"
        assert 1 in self.__t.neighbours(0), "connection destroyed"
        assert 0 in self.__t.neighbours(1), "connection not created"
        assert self.__t.isArc(1, 0), "No arc between 1 and 0"

    def testInDegreeAdd2Arcs2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.addArc(1, 0)
        assert self.__t.order() == 2, "Representation order is not 4"
        assert self.__t.size() == 2, "Representation size is not 1"
        assert 1 in self.__t.neighbours(0), "connection destroyed"
        assert 0 in self.__t.neighbours(1), "connection not created"
        assert self.__t.isArc(1, 0), "No arc between 1 and 0"

    def testNeighboursAdd2Arcs2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.addArc(1, 0)
        assert self.__t.order() == 2, "Representation order is not 4"
        assert self.__t.size() == 2, "Representation size is not 1"
        assert 1 in self.__t.neighbours(0), "connection destroyed"
        assert 0 in self.__t.neighbours(1), "connection not created"
        assert self.__t.isArc(1, 0), "No arc between 1 and 0"

    #
    def testOrderRemoveVertex1VertexRepresentation(self):
        assert self.__s.order() == 1, "Representation is not correct order"
        self.__s.removeVertex(0)
        assert self.__s.order() == 0, "Representation order is not 0"

    def testSizeRemoveVertex1VertexRepresentation(self):
        assert self.__s.size() == 0, "Representation is not correct size"
        self.__s.removeVertex(0)
        assert self.__s.size() == 0, "Representation size is not 0"

    #

    def testOrderRemove1Vertex2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        self.__t.removeVertex(1)
        assert self.__t.order() == 1, "Representation order is not 1"

    def testSizeRemove1Vertex2VertexRepresentation(self):
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.removeVertex(1)
        assert self.__t.size() == 0, "Representation size is not 0"

    def testSelfEdgesRemove1Vertex2VertexRepresentation(self):
        assert self.__t.selfEdges() == 0, "Representation should not have self edges"
        self.__t.removeVertex(1)
        assert self.__t.selfEdges() == 0, "Representation self edges is not 0"

    def testIsArcRemove1Vertex2VertexRepresentation(self):
        assert self.__t.isArc(0, 1), "Arc did not exist"
        self.__t.removeVertex(1)
        assert not self.__t.isArc(0, 0), "Arc wrongly persisted"

    def testDegreeRemove1Vertex2VertexRepresentation(self):
        self.__t.removeVertex(1)
        assert self.__t.degree(0) == 0, "Degree of 0 is not 0"

    def testInDegreeRemove1Vertex2VertexRepresentation(self):
        self.__t.removeVertex(1)
        assert self.__t.inDegree(0) == 0, "Degree of 0 is not 0"

    def testNeighboursRemove1Vertex2VertexRepresentation(self):
        self.__t.removeVertex(1)
        assert self.__t.neighbours(0) == [], "0 should have no neighbours"

    #
    def testOrderRemove2Vertex2VertexRepresentation(self):
        assert self.__t.order() == 2, "Representation is not correct order"
        self.__t.removeVertex(0)
        assert self.__t.order() == 1, "Representation order is not 1"
        self.__t.removeVertex(0)  # Relabelling
        assert self.__t.order() == 0, "Representation order is not 0"

    def testSizeRemove2Vertex2VertexRepresentation(self):
        assert self.__t.size() == 1, "Representation is not correct size"
        self.__t.removeVertex(0)
        assert self.__t.size() == 0, "Representation size is not 0"
        self.__t.removeVertex(0)  # Relabelling
        assert self.__t.size() == 0, "Representation size is not 0"

    #
    def testOrderRemove1VertexArcRelabelling(self):
        assert self.__u.order() == 5, "Representation is not correct order"
        self.__u.removeVertex(0)
        assert self.__u.order() == 4, "Representation order is not 4"

    def testSizeRemove1VertexArcRelabelling(self):
        assert self.__u.size() == 17, "Representation is not correct size"
        self.__u.removeVertex(0)
        assert self.__u.size() == 8, "Representation size is not 8"

    def testIsArcRemove1VertexArcRelabelling(self):
        self.__u.removeVertex(0)
        assert self.__u.isArc(1, 3), "2 divides 4 arc was not relabelled"
        assert not self.__u.isArc(1, 2), "2 divides 3 arc was fabricated in relabelling"
        for i in range(4):
            assert self.__u.isArc(0, i), "1 divides %s arc was not relabelled" % i

    def testDegreeRemove1VertexArcRelabelling(self):
        self.__u.removeVertex(0)
        for i, j in zip(list(range(4)), (4, 2, 1, 1)):
            assert self.__u.degree(i) == j, "degree of %s  was not %s" % (i, j)

    def testInDegreeRemove1VertexArcRelabelling(self):
        self.__u.removeVertex(0)
        for i, j in zip(list(range(4)), (1, 2, 2, 3)):
            assert self.__u.inDegree(i) == j, "Indegree of %s  was not %s" % (i, j)

    def testNeighboursRemove1VertexArcRelabelling(self):
        self.__u.removeVertex(0)
        for i, j in zip(list(range(4)), ([0, 1, 2, 3], [1, 3], [2], [3])):
            assert self.__u.neighbours(i) == j, "neighbours of %s  was not %s" % (i, j)

    #
    def testOrderRemoveMiddleVertexArcRelabelling(self):
        assert self.__u.order() == 5, "Representation is not correct order"
        self.__u.removeVertex(3)
        assert self.__u.order() == 4, "Representation order is not 4"

    def testSizeRemoveMiddleVertexArcRelabelling(self):
        assert self.__u.size() == 17, "Representation is not correct size"
        self.__u.removeVertex(3)
        assert self.__u.size() == 13, "Representation size is not 13"

    def testIsArcRemoveMiddleVertexArcRelabelling(self):
        self.__u.removeVertex(3)
        assert self.__u.isArc(2, 3), "2 divides 4 arc was not relabelled"

    def testDegreeRemoveMiddleVertexArcRelabelling(self):
        self.__u.removeVertex(3)
        for i, j in zip(list(range(4)), (4, 4, 3, 2)):
            assert self.__u.degree(i) == j, "degree of %s  was not %s" % (i, j)

    def testInDegreeRemoveMiddleVertexArcRelabelling(self):
        self.__u.removeVertex(3)
        for i, j in zip(list(range(4)), (4, 2, 3, 4)):
            assert self.__u.inDegree(i) == j, "Indegree of %s  was not %s" % (i, j)

    def testNeighboursRemoveMiddleVertexArcRelabelling(self):
        self.__u.removeVertex(3)
        for i, j in zip(list(range(4)), ([0, 1, 2, 3], [0, 1, 2, 3], [0, 2, 3], [0, 3])):
            assert self.__u.neighbours(i) == j, "neighbours of %s  was %s not %s" % (i, self.__u.neighbours(i), j)

    def testReadingLeavesNextLinesIntact(self):
        s = self.newRepresentation()
        stream = StreamMocker("2", "1", "2", "Mary had a little lamb", "42")
        s.read(stream, lambda x, y: None)  # Discarding the links formed
        assert stream.readline() == "Mary had a little lamb"
        assert stream.readline() == "42"
        self.assertRaises(StopIteration, stream.readline)


class ListTests(unittest.TestCase, EmptyTests, StandardTests):
    def newRepresentation(self):
        return AdjacencyLists()

    def setUp(self):
        EmptyTests.setUp(self)
        StandardTests.setUp(self)

        self.__t = self.newRepresentation()
        self.__t.addVertices

    def testReadingEmpty(self):
        """
        Cannot rely on empty list-based and matrix-based graphs being similar
        """
        s = self.newRepresentation()
        s.read(StreamMocker("0"), s.addArc)
        # WARNING!
        # This call relies on the fact that Representation.addArc
        # and Graph.addArc have the same signature, and that the latter
        # should by definition call the former.  If this arrangement were to
        # change, testing representation in isolation would be harder
        assert s.order() == 0, "Wrong order"
        assert s.size() == 0, "Wrong size"

    def testReading1V0A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("1", ""), s.addArc)
        assert s.order() == 1, "Wrong order"
        assert s.size() == 0, "Wrong size"
        assert not s.isArc(0, 0), "False connection"

    def testReading1V1A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("1", "0  "), s.addArc)
        assert s.order() == 1, "Wrong order"
        assert s.size() == 1, "Wrong size"
        assert s.isArc(0, 0), "Connection between 0, 0 missing"

    def testReading2V1A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("2", "1  ", ""), s.addArc)
        assert s.order() == 2, "Wrong order"
        assert s.size() == 1, "Wrong size"
        assert s.isArc(0, 1), "Connection between 0, 1 missing"

    def testReading2V2A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("2", "1 0", ""), s.addArc)
        assert s.order() == 2, "Wrong order"
        assert s.size() == 2, "Wrong size"
        assert s.isArc(0, 1), "Connection between 0, 1 missing"
        assert s.isArc(0, 0), "Connection between 0, 0 missing"

    def testReading5V3A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("5", "1", "2", "", "4", ""), s.addArc)
        assert s.order() == 5, "Wrong order"
        assert s.size() == 3, "Wrong size"
        assert s.isArc(0, 1), "Connection between 0, 1 missing"
        assert s.isArc(1, 2), "Connection between 1, 2 missing"
        assert s.isArc(3, 4), "Connection between 3, 4 missing"
        assert not s.isArc(2, 3), "Arc shouldn't exist"

    def testReading5VFullyConnected(self):
        s = self.newRepresentation()
        s.read(StreamMocker("5", "0 1 2 3 4", "0 1 2 3 4", "0 1 2 3 4", "0 1 2 3 4", "0 1 2 3 4"), s.addArc)
        assert s.order() == 5, "Wrong order"
        assert s.size() == 25, "Wrong size"
        for i in range(5):
            for j in range(5):
                assert s.isArc(i, j)

    def testWriting1V0E(self):
        r = self.newRepresentation()
        r.addVertices(1)
        assert str(r) == "1\n\n"

    def testWriting1V1E(self):
        r = self.newRepresentation()
        r.addVertices(1)
        r.addArc(0, 0)
        assert str(r) == "1\n0\n"

    def testWriting5V5A(self):
        r = self.newRepresentation()
        r.addVertices(5)
        for i in range(5):
            r.addArc(i, (i + 2) % 5)
        assert str(r) == "5\n2\n3\n4\n0\n1\n"


class MatrixTests(unittest.TestCase, EmptyTests, StandardTests):
    def newRepresentation(self):
        return AdjacencyMatrix()

    def setUp(self):
        EmptyTests.setUp(self)
        StandardTests.setUp(self)

    def testReadingEmpty(self):
        """
        Cannot rely on empty list-based and matrix-based graphs being similar
        """
        s = self.newRepresentation()
        s.read(StreamMocker("0"), s.addArc)
        assert s.order() == 0, "Wrong order"
        assert s.size() == 0, "Wrong size"

    def testReading1V0A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("1", "0"), s.addArc)
        assert s.order() == 1, "Wrong order"
        assert s.size() == 0, "Wrong size"
        assert not s.isArc(0, 0), "False connection"

    def testReading1V1A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("1", "1  "), s.addArc)
        assert s.order() == 1, "Wrong order"
        assert s.size() == 1, "Wrong size"
        assert s.isArc(0, 0), "Connection between 0, 0 missing"

    def testReading2V1A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("2", "0 1 ", "0 0 "), s.addArc)
        assert s.order() == 2, "Wrong order"
        assert s.size() == 1, "Wrong size"
        assert s.isArc(0, 1), "Connection between 0, 1 missing"

    def testReading2V2A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("2", "1 1", "0 0"), s.addArc)
        assert s.order() == 2, "Wrong order"
        assert s.size() == 2, "Wrong size"
        assert s.isArc(0, 1), "Connection between 0, 1 missing"
        assert s.isArc(0, 0), "Connection between 0, 0 missing"

    def testReading5V3A(self):
        s = self.newRepresentation()
        s.read(StreamMocker("5", "0 1 0 0 0", "0 0 1 0 0", "0 0 0 0 0", "0 0 0 0 1  ", "0 0 0 0 0"), s.addArc)
        assert s.order() == 5, "Wrong order"
        assert s.size() == 3, "Wrong size"
        assert s.isArc(0, 1), "Connection between 0, 1 missing"
        assert s.isArc(1, 2), "Connection between 1, 2 missing"
        assert s.isArc(3, 4), "Connection between 3, 4 missing"
        assert not s.isArc(2, 3), "Arc shouldn't exist"

    def testReading5VFullyConnected(self):
        s = self.newRepresentation()
        s.read(StreamMocker("5", "1 1 1 1 1", "1 1 1 1 1", "1 1 1 1 1", "1 1 1 1 1", "1 1 1 1 1"), s.addArc)
        assert s.order() == 5, "Wrong order"
        assert s.size() == 25, "Wrong size"
        for i in range(5):
            for j in range(5):
                assert s.isArc(i, j)

    def testWriting1V0E(self):
        r = self.newRepresentation()
        r.addVertices(1)
        assert str(r) == "1\n0\n"

    def testWriting1V1E(self):
        r = self.newRepresentation()
        r.addVertices(1)
        r.addArc(0, 0)
        assert str(r) == "1\n1\n"

    def testWriting5V5A(self):
        r = self.newRepresentation()
        r.addVertices(5)
        for i in range(5):
            r.addArc(i, (i + 2) % 5)
        assert str(r) == "5\n0 0 1 0 0\n0 0 0 1 0\n0 0 0 0 1\n1 0 0 0 0\n0 1 0 0 0\n"


if __name__ == "__main__":
    unittest.main()
