"""This suite of tests is intended to test the correctness of graph
implementations within the graphtypes module.
"""
from io import StringIO
import unittest

from testing.testutils import StreamMocker, TestStub

from graphadt.graphtypes import Graph, DirectedGraph, UndirectedGraph



def suite():
    """Returns a suite of all TestCases defined in this module
    """
    # Load all the tests from this module
    return unittest.TestLoader().loadTestsFromName(__name__)


class AbstractTest(object):
    """These tests check that all method calls on Graph object have the right behaviour.
    """

    def testAddVertices(self):
        self.g.addVertices(1)
        assert self.r._log == ["addVertices(1,)"]

    def testRemoveVertex(self):
        self.g.removeVertex(0)
        assert self.r._log == ["removeVertex(0,)"]

    def testAddEdge(self):
        self.g.addEdge(0, 1)
        assert self.r._log == ["addArc(0, 1)", "addArc(1, 0)"]

    def testRemoveEdge(self):
        self.g.removeEdge(0, 1)
        assert self.r._log == ["removeArc(0, 1)", "removeArc(1, 0)"]

    def testIsEdge(self):
        assert self.g.isEdge(0, 1)
        assert self.r._log == ["isArc(0, 1)", "isArc(1, 0)"]

    def testDegree(self):
        assert self.g.degree(0) == 5
        assert self.r._log == ["degree(0,)"]

    def testNeighbours(self):
        assert self.g.neighbours(0) == 5
        assert self.r._log == ["neighbours(0,)"]

    def testOrder(self):
        assert self.g.order() == 5
        assert self.r._log == ["order()"]

    def testRead(self):
        s = StreamMocker("3", "2", "1", "0")

        class TestGraph(Graph):
            defaultConnector = "default Connector method"
            inDegree = size = None

            def __init__(self):
                Graph.__init__(self, TestStub())

        readGraph = TestGraph.read(s)
        prediction = ["read(" + str(s) + ", 'default Connector method')"]
        assert readGraph._representation._log == prediction
        assert s.readline() == "3", "Shouldn't have read any lines"

    def testStr(self):
        s = str(self.g)
        assert s == str(self.r)
        assert self.r._log == []  # str(self.r) is a method call on object, not on Mocker, and so will not be logged.

    def testCopy(self):
        class TestStubGraph(Graph, TestStub):
            defaultConnector = inDegree = size = None

            def __init__(self):
                Graph.__init__(self, TestStub())
                TestStub.__init__(self)

        r = TestStub([4, [1], [1, 2], [2, 3, 4], [1, 3, 4]])
        self.g._representation = r  # Set the internal representation of g to be what we want it to be
        copy = TestStubGraph.copy(self.g)
        prediction = [
            "addVertices(4,)",
            "defaultConnector(0, 1)",
            "defaultConnector(1, 1)",
            "defaultConnector(1, 2)",
            "defaultConnector(2, 2)",
            "defaultConnector(2, 3)",
            "defaultConnector(2, 4)",
            "defaultConnector(3, 1)",
            "defaultConnector(3, 3)",
            "defaultConnector(3, 4)",
        ]
        assert copy._log == prediction, "Wrong state detected"


class DirectedTests(unittest.TestCase, AbstractTest):
    def setUp(self):
        self.r = TestStub()
        self.g = DirectedGraph(self.r)

    def testAddArc(self):
        self.g.addArc(0, 1)
        assert self.r._log == ["addArc(0, 1)"]

    def testRemoveArc(self):
        self.g.removeArc(0, 1)
        assert self.r._log == ["removeArc(0, 1)"]

    def testIsArc(self):
        assert self.g.isArc(0, 1) == 5
        assert self.r._log == ["isArc(0, 1)"]

    def testInDegree(self):
        assert self.g.inDegree(0) == 5
        assert self.r._log == ["inDegree(0,)"]

    def testSize(self):
        assert self.g.size() == 5
        assert self.r._log == ["size()"]


class UndirectedTests(unittest.TestCase, AbstractTest):
    def setUp(self):
        self.r = TestStub()
        self.g = UndirectedGraph(self.r)  # Default constructor, leaves graph empty

    def testInDegree(self):
        assert self.g.inDegree(0) == 5
        assert self.r._log == ["degree(0,)"]

    def testSize(self):
        self.r = TestStub([5, 3])
        self.g = UndirectedGraph(self.r)
        assert self.g.size() == 4
        assert self.r._log == ["size()", "selfEdges()"]


if __name__ == "__main__":
    unittest.main()
