"""This suite of tests is intended to test the correctness of algorithm
implementations within the graphtypes module.
"""

from graphadt.graphtypes import (
    UndirectedGraph,
    DirectedAdjListsGraph,
    UndirectedAdjListsGraph,
    DirectedAdjMatrixGraph,
    UndirectedAdjMatrixGraph,
)
from graphadt.algorithms import (
    getBFSParents,
    getDFSParents,
    BFS,
    DFS,
    isConnected,
    isStronglyConnected,
    strongComponents,
    isAcyclic,
    girth,
    isBipartite,
    maxDistance,
    distanceMatrix,
    diameter,
    radius,
    topSort1,
    topSort2,
)

import unittest
import math


def suite():
    return unittest.TestLoader().loadTestsFromName(__name__)


class GetBFSParentsTests(object):
    def testBFSParentsEmptyGraph(self):
        g = self.getEmptyGraph()
        self.assertRaises(ValueError, getBFSParents, g, 0)
        self.assertRaises(ValueError, getBFSParents, g, -1)
        self.assertRaises(ValueError, getBFSParents, g, 1)

    def testBFSParents1V0EGraph(self):
        g = self.get1V0EGraph()
        assert getBFSParents(g, 0) == [0], "Wrong parents returned"
        self.assertRaises(ValueError, getBFSParents, g, -1)
        self.assertRaises(ValueError, getBFSParents, g, 1)

    def testBFSParents1V1EGraph(self):
        g = self.get1V1EGraph()
        assert getBFSParents(g, 0) == [0], "Wrong parents returned"

    def testBFSParents2V1EGraph(self):
        g = self.get2V1EGraph()
        assert getBFSParents(g, 0) == [0, 0], "Wrong parents returned"

    def testBFSParentsTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert getBFSParents(g, 4) == [2, 0, 4, 2, 4, 4, 5, 6], "Wrong parents returned"

    def testBFSParentsTreeGraphWithLoops(self):
        g = self.getUndirectedTreeGraphWithLoops()
        assert getBFSParents(g, 4) == [2, 6, 4, 2, 4, 4, 4, 6], "Wrong parents returned"

    def testBFSParentsTreeGraphUnconnected(self):
        g = self.getUndirectedTreeGraphUnconnected()
        assert getBFSParents(g, 4) == [2, 0, 4, 2, 4, None, None, None], "Wrong parents returned"


class BFSTests(object):
    def testBFSEmptyGraph(self):
        g = self.getEmptyGraph()
        self.assertRaises(ValueError, BFS, g, 0)
        self.assertRaises(ValueError, BFS, g, -1)
        self.assertRaises(ValueError, BFS, g, 1)

    def testBFS1V0EGraph(self):
        g = self.get1V0EGraph()
        level, levelorder = BFS(g, 0)
        assert level == 1, "Wrong level returned"
        assert levelorder == [0], "Levelorder was wrong"
        self.assertRaises(ValueError, getBFSParents, g, -1)
        self.assertRaises(ValueError, getBFSParents, g, 1)

    def testBFS1V1EGraph(self):
        g = self.get1V1EGraph()
        level, levelorder = BFS(g, 0)
        assert level == 1, "Wrong level"
        assert levelorder == [0], "Wrong levelorder"

    def testBFSTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        count, levels = BFS(g, 4)
        assert count == 8, "Wrong number of vertices traversed"
        assert levels == [3, 6, 1, 4, 0, 2, 5, 7], "Wrong level order returned"

    def testBFSTreeGraphWithLoops(self):
        g = self.getUndirectedTreeGraphWithLoops()
        count, levels = BFS(g, 4)
        assert count == 8, "Wrong number of vertices detected"
        assert levels == [4, 6, 1, 5, 0, 2, 3, 7], "Wrong level order returned"

    def testBFSParentsTreeGraphUnconnected(self):
        g = self.getUndirectedTreeGraphUnconnected()
        count, levels = BFS(g, 4)
        assert count == 5, "Wrong number of vertices detected"
        assert levels == [2, 4, 1, 3, 0, None, None, None], "Wrong level order returned"


class DFSTests(object):
    def testDFSEmptyGraph(self):
        g = self.getEmptyGraph()
        self.assertRaises(ValueError, DFS, g, 0)
        self.assertRaises(ValueError, DFS, g, -1)
        self.assertRaises(ValueError, DFS, g, 1)

    def testDFS1V0EGraph(self):
        g = self.get1V0EGraph()
        count, preorder, postorder = DFS(g, 0)
        assert count == 1, "Wrong number of vertices"
        assert preorder == [1], "Preorder was wrong"
        assert postorder == [1], "Post order was wrong"
        self.assertRaises(ValueError, getBFSParents, g, -1)
        self.assertRaises(ValueError, getBFSParents, g, 1)

    def testDFS1V1EGraph(self):
        g = self.get1V1EGraph()
        count, preorder, postorder = DFS(g, 0)
        assert count == 1, "Wrong number of vertices"
        assert preorder == [1], "Preorder was wrong"
        assert postorder == [1], "Post order was wrong"

    def testDFSTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        count, preorder, postorder = DFS(g, 4)
        assert count == 8, "Wrong number of vertices traversed"
        assert preorder == [3, 4, 2, 5, 1, 6, 7, 8], "Wrong preorder returned"
        assert postorder == [2, 1, 4, 3, 8, 7, 6, 5], "Wrong postorder returned"

    def testDFSTreeGraphWithLoops(self):
        g = self.getUndirectedTreeGraphWithLoops()
        count, preorder, postorder = DFS(g, 4)
        assert count == 8, "Wrong number of vertices detected"
        assert preorder == [3, 4, 2, 8, 1, 6, 5, 7], "Wrong preorder returned"
        assert postorder == [5, 4, 7, 6, 8, 1, 3, 2], "Wrong postorder returned"

    def testDFSTreeGraphUnconnected(self):
        g = self.getUndirectedTreeGraphUnconnected()
        count, preorder, postorder = DFS(g, 4)
        assert count == 5, "Wrong number of vertices detected"
        assert preorder == [3, 4, 2, 5, 1, None, None, None], "Wrong preorder returned"
        assert postorder == [2, 1, 4, 3, 5, None, None, None], "Wrong postorder returned"


class IsConnectedTests(object):
    def testIsConnectedEmptyGraph(self):
        g = self.getEmptyGraph()
        assert isConnected(g), "An empty graph should be trivially connected"

    def testIsConnected1V0EGraph(self):
        g = self.get1V0EGraph()
        assert isConnected(g), "A vertex should be connected to itself implicity."

    def testDFS1V1EGraph(self):
        g = self.get1V1EGraph()
        assert isConnected(g), "This vertex is definitely connected to itself."

    def testIsConnectedUndirectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert isConnected(g), "This graph is connected"

    def testIsConnectedUndirectedTreeGraphUnconnected(self):
        g = self.getUndirectedTreeGraphUnconnected()
        assert not isConnected(g), "This graph is not connected"


class IsStronglyConnectedTests(object):
    def testIsStronglyConnectedEmptyGraph(self):
        g = self.getEmptyGraph()
        assert isConnected(g), "An empty graph should be trivially strongly connected"

    def testIsStronglyConnected1V0EGraph(self):
        g = self.get1V0EGraph()
        assert isStronglyConnected(g), "A vertex should be strongly connected to itself implicity."

    def testIsStronglyConnected1V1EGraph(self):
        g = self.get1V1EGraph()
        assert isStronglyConnected(g), "This vertex is definitely strongly connected to itself."

    def testIsStronglyConnected2V0EGraph(self):
        g = self.get2V0EGraph()
        assert not isStronglyConnected(g), "This graph is not strongly connected."

    def testIsStronglyConnectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert isStronglyConnected(g), "This graph is strongly connected."

    def testIsStronglyConnectedTreeGraphUnconnected(self):
        g = self.getUndirectedTreeGraphUnconnected()
        assert not isStronglyConnected(g), "This graph is not connected at all."


class StrongComponentsTests(object):
    def testStrongComponentsEmptyGraph(self):
        g = self.getEmptyGraph()
        assert strongComponents(g) == [], "No vertices implies no strongly connected components"

    def testStrongComponents1V0EGraph(self):
        g = self.get1V0EGraph()
        assert strongComponents(g) == [0], "A vertex should be strongly connected to itself implicity."

    def testStrongComponents1V1EGraph(self):
        g = self.get1V1EGraph()
        assert strongComponents(g) == [0], "This vertex is definitely strongly connected to itself."

    def testStrongComponents2V0EGraph(self):
        g = self.get2V0EGraph()
        assert strongComponents(g) == [0, 1], "Each vertex is its own strongly connected component!"

    def testStrongComponents2V1EGraph(self):
        g = self.get2V1EGraph()
        assert strongComponents(g) == [0, 0], "Both vertices are in the one strongly connected component"


class GetDFSParentsTests(object):
    def testGetDFSParentsEmptyGraph(self):
        g = self.getEmptyGraph()
        func = getDFSParents
        self.assertRaises(ValueError, func, g, 0)
        self.assertRaises(ValueError, func, g, -1)
        self.assertRaises(ValueError, func, g, 1)

    def testGetDFSParents1V0EGraph(self):
        g = self.get1V0EGraph()
        assert getDFSParents(g, 0) == [0], "Zero should be its own parent"

    def testGetDFSParents1V1EGraph(self):
        g = self.get1V1EGraph()
        assert getDFSParents(g, 0) == [0], "Zero should still be its own parent"

    def testGetDFSParents2V1EGraph(self):
        g = self.get2V1EGraph()
        assert getDFSParents(g, 0) == [0, 0], "Zero is parent of both"
        assert getDFSParents(g, 1) == [1, 1], "One is parent of both"

    def testGetDFSParentsUndirectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert getDFSParents(g, 4) == [2, 0, 4, 2, 4, 4, 5, 6], "Wrong parents returned"
        assert getDFSParents(g, 1) == [1, 1, 0, 2, 2, 4, 5, 6], "Wrong parents returned"


class IsAcyclicTests(object):
    def testIsAcyclicEmptyGraph(self):
        g = self.getEmptyGraph()
        assert isAcyclic(g), "An empty graph is trivially acyclic"

    def testIsAcyclic1V0EGraph(self):
        g = self.get1V0EGraph()
        assert isAcyclic(g), "One vertex, no edge, is acyclic"

    def testIsAcyclic1V1EGraph(self):
        g = self.get1V1EGraph()
        if isinstance(g, UndirectedGraph):
            assert not isAcyclic(g), "One vertex, one edge, must be cyclic"
        else:
            assert not isAcyclic(g), "Directed graphs with edges are cyclic"

    def testIsAcyclic2V1EGraph(self):
        g = self.get2V1EGraph()
        if isinstance(g, UndirectedGraph):
            assert isAcyclic(g), "This graph has no cycles"
        else:
            assert not isAcyclic(g), "In a directed graph, edges imply cycles"

    def testIsAcyclicUndirectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        if isinstance(g, UndirectedGraph):
            assert isAcyclic(g), "A tree by definition has no cycles"
        else:
            assert not isAcyclic(g), "In a directed graph, the edges imply cycles"

    def testIsAcyclicUndirectedTreeGraphWithLoops(self):
        g = self.getUndirectedTreeGraphWithLoops()
        assert not isAcyclic(g), "Once a graph has had loops added, it is no longer a tree"


class GirthTests(object):
    def testGirthEmptyGraph(self):
        g = self.getEmptyGraph()
        assert math.isinf(girth(g)), "An empty graph should have girth infinity"

    def testGirth1V1EGraph(self):
        g = self.get1V1EGraph()
        assert girth(g) == 1, "Self loops return girth of 1"

    def testGirth2V1EGraph(self):
        g = self.get2V1EGraph()
        assert math.isinf(girth(g)), "You can't come back along the same edge"

    def testGirthTriangle(self):
        g = self.get2V1EGraph()
        g.addVertices(2)
        g.addEdge(1, 2)
        g.addEdge(2, 3)
        g.addEdge(3, 1)
        assert girth(g) == 3, "This graph has a triangle in it"

    def testGirthHangingTriangle(self):
        g = self.get2V1EGraph()
        g.addVertices(8)
        for i in range(9):
            g.addEdge(i, i + 1)
        g.addEdge(9, 7)
        assert girth(g) == 3, "This graph also has a triangle in it, down a long path"

    def testGirthPentagraph(self):
        g = self.getGraphInstance()
        g.addVertices(5)
        for i in range(5):
            g.addEdge(i, (i + 2) % 5)
        assert girth(g) == 5, "It's a real star!"


class IsBipartiteTests(object):
    def testIsBipartiteEmptyGraph(self):
        g = self.getEmptyGraph()
        assert isBipartite(g), "An empty graph is trivially bipartite"

    def testIsBipartite1V0EGraph(self):
        g = self.get1V0EGraph()
        assert isBipartite(g), "The vertex can be coloured arbitrarily"

    def testIsBipartite1V1EGraph(self):
        g = self.get1V1EGraph()
        assert not isBipartite(g), "Self-loops violate the colouring condition\n" + "A vertex cannot be two colours at once"

    def testIsBipartite2V1EGraph(self):
        g = self.get2V1EGraph()
        assert isBipartite(g), "Should be bipartite"

    def testIsBipartiteTriangleGraph(self):
        g = self.getGraphInstance()
        g.addVertices(3)
        g.addEdge(0, 1)
        g.addEdge(1, 2)
        g.addEdge(0, 2)
        assert not isBipartite(g), "Triangles cannot be bipartite"

    def testIsBipartitePentagram(self):
        g = self.getGraphInstance()
        g.addVertices(5)
        for i in range(5):
            g.addEdge(i, (i + 2) % 5)
        assert not isBipartite(g), "Cycle with odd number of vertices cannot be bipartite"

    def testIsBipartiteHexagram(self):
        g = self.getGraphInstance()
        g.addVertices(6)
        for i in range(6):
            g.addEdge(i, (i + 2) % 6)
        assert not isBipartite(g), "Two triangles are not bipartite"

    def testIsBipartiteOctogram(self):
        g = self.getGraphInstance()
        g.addVertices(8)
        for i in range(8):
            g.addEdge(i, (i + 2) % 8)
        assert isBipartite(g), "Two squares are both bipartite"

    def testIsBipartiteTreeUnconnected(self):
        g = self.getUndirectedTreeGraphUnconnected()
        assert isBipartite(g), "This tree should be bipartite"


# These following tests are fundamentally different depending on the type of graph called on
# And so are separated in DirectedTests and UndirectedTests

# class TopSort1Tests(object):
# class TopSort2Tests(object):


class MaxDistanceTests(object):
    def testMaxDistanceEmptyGraph(self):
        g = self.getEmptyGraph()
        func = maxDistance
        self.assertRaises(ValueError, func, g, 0)
        self.assertRaises(ValueError, func, g, -1)
        self.assertRaises(ValueError, func, g, 1)

    def testMaxDistance1V0EGraph(self):
        g = self.get1V0EGraph()
        assert maxDistance(g, 0)[0] == 0, "Wrong maximum distance returned"

    def testMaxDistance1V1EGraph(self):
        g = self.get1V1EGraph()  # Potential to loop infinitely
        assert maxDistance(g, 0)[0] == 0, "Length of maximum length cycle-free path"

    def testMaxDistance2V1EGraph(self):
        g = self.get2V1EGraph()
        assert maxDistance(g, 0)[0] == 1, "Wrong length returned"
        assert maxDistance(g, 1)[0] == 1, "Wrong length returned"

    def testMaxDistancePentaGraph(self):
        g = self.getEmptyGraph()
        g.addVertices(5)
        for i in range(5):
            g.addEdge(i, (i + 2) % 5)
        assert maxDistance(g, 0)[0] == 2, "The furthest vertex is 2 steps away from the start"

    def testMaxDistanceLongChain(self):
        g = self.getEmptyGraph()
        g.addVertices(10)
        for i in range(9):
            g.addEdge(i, (i + 1) % 10)
        d, l = maxDistance(g, 0)
        assert d == 9, "The furthest vertex is 9 steps away from the start"
        assert l == [x for x in range(10)], "Each vertex is one further than the previous"

    def testMaxDistanceUndirectedTreeGraph(self):
        # More elaborate version of above
        g = self.getUndirectedTreeGraph()
        d, l = maxDistance(g, 4)
        assert d == 3, "Maximum distance is 3"
        assert l == [2, 3, 1, 2, 0, 1, 2, 3], "Wrong distance graph returned"

    def testMaxDistanceUnreachableNodes(self):
        g = self.getEmptyGraph()
        g.addVertices(2)
        d, l = maxDistance(g, 0)
        assert math.isinf(d), "Maximum distance is by convention labelled infinity"
        assert l == [0, float("infinity")], "A vertex which cannot be reached has infinite distance"

    def testMaxDistanceUndirectedTreeGraphUnconnected(self):
        # More elaborate version of above
        g = self.getUndirectedTreeGraphUnconnected()
        d, l = maxDistance(g, 4)
        assert math.isinf(d), "Maximum distance is by convention labelled infinity"
        assert l == [2, 3, 1, 2, 0, float("inf"), float("inf"), float("inf")], "A vertex which cannot be reached has infinite distance"


class DistanceMatrixTests(object):
    def testDistanceMatrixEmptyGraph(self):
        g = self.getEmptyGraph()
        assert distanceMatrix(g) == [], "Empty graph should return empty list"

    def testDistanceMatrix1V0EGraph(self):
        g = self.get1V0EGraph()
        assert distanceMatrix(g) == [[0]], "No distance from itself to itself"

    def testDistanceMatrix1V1EGraph(self):
        g = self.get1V1EGraph()
        assert distanceMatrix(g) == [[0]], "No distance from itself to itself"

    def testDistanceMatrix2V1EGraph(self):
        g = self.get2V1EGraph()
        assert distanceMatrix(g) == [[0, 1], [1, 0]], "Symmetric matrix of distance 1 to the other"

    def testDistanceMatrix2V0EGraph(self):
        g = self.get2V0EGraph()
        inf = float("inf")
        assert distanceMatrix(g) == [[0, inf], [inf, 0]], "Expect to see infinite distances"

    def testDistanceMatrixUndirectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert distanceMatrix(g) == [
            [0, 1, 1, 2, 2, 3, 4, 5],
            [1, 0, 2, 3, 3, 4, 5, 6],
            [1, 2, 0, 1, 1, 2, 3, 4],
            [2, 3, 1, 0, 2, 3, 4, 5],
            [2, 3, 1, 2, 0, 1, 2, 3],
            [3, 4, 2, 3, 1, 0, 1, 2],
            [4, 5, 3, 4, 2, 1, 0, 1],
            [5, 6, 4, 5, 3, 2, 1, 0],
        ], "That is one huge matrix"


class DiameterTests(object):
    def testDiameterEmptyGraph(self):
        g = self.getEmptyGraph()
        assert diameter(g) == 0, "An empty graph has diameter 0"

    def testDiameter1V0EGraph(self):
        g = self.get1V0EGraph()
        assert diameter(g) == 0, "With one vertex, diameter is 0"

    def testDiameter1V1EGraph(self):
        g = self.get1V1EGraph()
        assert diameter(g) == 0, "Don't be fooled by the selfarc"

    def testDiameter2V1EGraph(self):
        g = self.get2V1EGraph()
        assert diameter(g) == 1, "Diameter should be one"

    def testDiameter2V0EGraph(self):
        g = self.get2V0EGraph()
        assert math.isinf(diameter(g)), "Diameter of infinity"

    def testDiameterUnconnectedWithConnectedSubcomponentsGraph(self):
        g = self.get2V0EGraph()
        g.addVertices(1)
        g.addEdge(1, 2)
        assert math.isinf(diameter(g)), "Radius of one"

    def testDiameterUndirectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert diameter(g) == 6, "Diameter of 6"


class RadiusTests(object):
    def testRadiusEmptyGraph(self):
        g = self.getEmptyGraph()
        assert radius(g) == 0, "An empty graph has radius 0"

    def testRadius1V0EGraph(self):
        g = self.get1V0EGraph()
        assert radius(g) == 0, "With one vertex, radius is 0"

    def testRadius1V1EGraph(self):
        g = self.get1V1EGraph()
        assert radius(g) == 0, "Don't be fooled by the selfarc"

    def testRadius2V1EGraph(self):
        g = self.get2V1EGraph()
        assert radius(g) == 1, "Radius should be one"

    def testRadius2V0EGraph(self):
        g = self.get2V0EGraph()
        assert math.isinf(radius(g)), "Radius of infinity"

    def testRadiusUnconnectedWithConnectedSubcomponentsGraph(self):
        g = self.get2V0EGraph()
        g.addVertices(1)
        g.addEdge(1, 2)
        assert math.isinf(radius(g)), "Unconnected means infinite radius"

    def testRadiusUndirectedTreeGraph(self):
        g = self.getUndirectedTreeGraph()
        assert radius(g) == 3, "Radius of 3"


class AlgorithmTests(
    GetBFSParentsTests,
    BFSTests,
    DFSTests,
    IsConnectedTests,
    IsStronglyConnectedTests,
    StrongComponentsTests,
    GetDFSParentsTests,
    IsAcyclicTests,
    GirthTests,
    IsBipartiteTests,
    MaxDistanceTests,
    DistanceMatrixTests,
    DiameterTests,
    RadiusTests,
):
    pass


class GraphSetups(object):
    """
    This class contains code used by any graphs in setting up their
    fixtures
    """

    def getEmptyGraph(self):
        g = self.getGraphInstance()
        return g

    def get1V0EGraph(self):
        g = self.getGraphInstance()
        g.addVertices(1)
        return g

    def get1V1EGraph(self):
        g = self.getGraphInstance()
        g.addVertices(1)
        g.addEdge(0, 0)
        return g

    def get2V0EGraph(self):
        g = self.getGraphInstance()
        g.addVertices(2)
        return g

    def get2V1EGraph(self):
        g = self.getGraphInstance()
        g.addVertices(2)
        g.addEdge(0, 1)
        return g

    def getUndirectedTreeGraph(self):
        """
        (4 (5 (6 (7)))
           (2 (0 (1))
              (3)))
        """
        g = self.getEmptyGraph()
        g.addVertices(8)
        g.addEdge(4, 5)
        g.addEdge(5, 6)
        g.addEdge(6, 7)
        g.addEdge(4, 2)
        g.addEdge(2, 0)
        g.addEdge(2, 3)
        g.addEdge(0, 1)
        return g

    def getUndirectedTreeGraphWithLoops(self):
        g = self.getUndirectedTreeGraph()
        g.addEdge(6, 4)
        g.addEdge(1, 6)
        return g

    def getUndirectedTreeGraphUnconnected(self):
        g = self.getUndirectedTreeGraph()
        g.removeEdge(4, 5)
        return g


class DirectedGraphTests(GraphSetups):
    """
    This class contains specific code meant to be used
    only if testing a directed graph
    """

    def get1V1AGraph(self):
        g = self.getGraphInstance()
        g.addVertices(1)
        g.addArc(0, 0)
        return g

    def get2V1AGraph(self):
        g = self.getGraphInstance()
        g.addVertices(2)
        g.addArc(0, 1)
        return g

    def getDirectedTreeGraph(self):
        """
        (4 (5 (6 (7)))
           (2 (0 (1))
              (3)))
        """
        g = self.getEmptyGraph()
        g.addVertices(8)
        g.addArc(4, 5)
        g.addArc(5, 6)
        g.addArc(6, 7)
        g.addArc(4, 2)
        g.addArc(2, 0)
        g.addArc(2, 3)
        g.addArc(0, 1)
        return g

    def getDirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraph()
        g.addArc(6, 4)
        g.addArc(1, 6)
        return g

    def getDirectedTreeGraphUnconnected(self):
        g = self.getDirectedTreeGraph()
        g.removeArc(4, 5)
        return g

    def testGetBFSParentsDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert getBFSParents(g, 4) == [2, 0, 4, 2, 4, 4, 5, 6], "Wrong parents returned"
        assert getBFSParents(g, 0) == [0, 0, None, None, None, None, None, None], "Wrong parents returned"

    def testBFSDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        count, levels = BFS(g, 4)
        assert count == 8
        assert levels == [3, 6, 1, 4, 0, 2, 5, 7], "Wrong levelorder returned"

    def testBFSDirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        count, levels = BFS(g, 4)
        assert count == 8
        assert levels == [3, 6, 1, 4, 0, 2, 5, 7], "Wrong levelorder returned"

    def testDFSDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        count, pre, post = DFS(g, 4)
        assert count == 8
        assert pre == [3, 4, 2, 5, 1, 6, 7, 8], "Wrong preorder returned"
        assert post == [2, 1, 4, 3, 8, 7, 6, 5], "Wrong postorder returned"

    def testIsConnectedDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert isConnected(g), "This directed graph is connected"

    def testIsConnectedDirectedTreeGraphUnconnected(self):
        g = self.getDirectedTreeGraphUnconnected()
        assert not isConnected(g), "This directed graph is not connected"

    def testIsConnected2V1A(self):
        g = self.get2V1AGraph()
        assert isConnected(g), "This directed graph is small, but connected"

    def testIsStronglyConnectedDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert not isStronglyConnected(g), "Directed Trees cannot be strongly connected"

    def testIsStronglyConnectedDirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        assert not isStronglyConnected(g), "This directed graph is no longer a tree, but is still not strongly connected"

    def testIsStronglyConnectedDirectedAdhocGraph(self):
        g = self.getEmptyGraph()
        g.addVertices(5)
        for i in range(5):
            g.addArc(i, (i + 2) % 5)
        assert isStronglyConnected(g), "This directed graph is a pentagram, and therefore strongly connected"

    def testStrongComponents2V1AGraph(self):
        g = self.get2V1AGraph()
        assert strongComponents(g) == [0, 1], "Each vertex is its own strongly connected component!"

    def testStroncComponentsDirectedAdhocGraph(self):
        g = self.getEmptyGraph()
        g.addVertices(5)
        for i in range(5):
            g.addArc(i, (i + 2) % 5)

        assert strongComponents(g) == [0] * 5, "Only one connected component "

    def testStrongComponentsDirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        assert strongComponents(g) == [0, 0, 0, 1, 0, 0, 0, 2], "Wrong strong components detected"

    def testGetDFSParentsDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert getDFSParents(g, 4) == [2, 0, 4, 2, 4, 4, 5, 6]

    def testGetDFSParentsDirectedTreeGraphUnconnected(self):
        g = self.getDirectedTreeGraphUnconnected()
        assert getDFSParents(g, 4) == [2, 0, 4, 2, 4, None, None, None]

    def testIsAcyclicAdhocGraph(self):
        g = self.getGraphInstance()
        g.addVertices(4)
        for i in range(1, 4):
            g.addArc(0, i)
        assert isAcyclic(g), "Something is very wrong here"

    def testIsAcyclicDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert isAcyclic(g), "A tree is, by definition, not cyclic"

    def testIsAcyclicDirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        assert not isAcyclic(g), "This erstwhile tree has had back-edges inserted"

    def testGirth2V1AGraph(self):
        g = self.get2V1AGraph()
        assert math.isinf(girth(g)), "The girth of this graph is infinity as there are no cycles"

    def testGirthDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert math.isinf(girth(g)), "The girth of this graph is infinity as there are no cycles"

    def testGirthDirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        assert girth(g) == 3, "The girth of this graph is infinity as there are no cycles"

    def testGirthDirectedPentagram(self):
        g = self.getGraphInstance()
        g.addVertices(5)
        for i in range(5):
            g.addArc(i, (i + 2) % 5)
        assert girth(g) == 5, "Girth of a pentagram is 5"

    def testIsBipartiteDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert isBipartite(g), "Trees are alwas Bipartite"

    def testIsBipartiteTwoLoopsArbitraryPlanning(self):
        g = self.getGraphInstance()
        g.addVertices(8)
        for i in range(4):
            # create two squares, [0, 1, 2, 3] and [4, 5, 6, 7]
            g.addArc(i, (i + 1) % 4)
            g.addArc(i + 4, (i + 1) % 4 + 4)
        g.addArc(7, 2)  # Connect the last odd vertex with an even one
        assert isBipartite(g), "Arbitrary choices can make colouring this instance harder"

    def testTopSort1EmptyGraph(self):
        g = self.getEmptyGraph()
        func = topSort1
        self.assertRaises(ValueError, func, g, 0)
        self.assertRaises(ValueError, func, g, -1)
        self.assertRaises(ValueError, func, g, 1)

    def testTopSort11V0EGraph(self):
        g = self.get1V0EGraph()
        assert topSort1(g, 0) == [0], "Wrong topological ordering returned"

    def testTopSort11V1EGraph(self):
        g = self.get1V1EGraph()
        self.assertRaises(ValueError, topSort1, g, 0), "Trying to topologically sort a cyclic graph" + " is dangerous"

    def testTopSort12V1EGraph(self):
        g = self.get2V1EGraph()
        self.assertRaises(ValueError, topSort1, g, 0), "Mutual dependencies cannot be topsorted"

    def testTopSort1DirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert topSort1(g, 4) == [6, 7, 4, 5, 0, 1, 2, 3], "Wrong topological ordering returned"

    def testTopSort1DirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        self.assertRaises(ValueError, topSort1, g, 1), "A cyclic graph cannot be sorted topologically"

    def testTopSort2EmptyGraph(self):
        g = self.getEmptyGraph()
        assert topSort2(g) == [], "Empty graph should return empty list"

    def testTopSort21V0EGraph(self):
        g = self.get1V0EGraph()
        assert topSort2(g) == [0], "Wrong topological ordering returned"

    def testTopSort21V1EGraph(self):
        g = self.get1V1EGraph()
        self.assertRaises(ValueError, topSort2, g), "Trying to topologically sort a cyclic graph" + " is dangerous"

    def testTopSort22V1EGraph(self):
        g = self.get2V1EGraph()
        self.assertRaises(ValueError, topSort2, g), "Mutual dependencies cannot be topsorted"

    def testTopSort2DirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert topSort2(g) == [6, 7, 4, 5, 0, 1, 2, 3], "Wrong topological ordering returned"

    def testTopSort2DirectedTreeGraphWithLoops(self):
        g = self.getDirectedTreeGraphWithLoops()
        self.assertRaises(ValueError, topSort2, g), "A cyclic graph cannot be sorted Topologically"

    def testMaxDistanceDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert maxDistance(g, 4)[0] == 3, "There are still two paths of length 3"

    def testMaxDistanceDirectedDecisions(self):
        g = self.getEmptyGraph()
        g.addVertices(5)
        g.addArc(0, 1)
        g.addArc(1, 2)
        g.addEdge(2, 3)
        g.addEdge(3, 4)
        g.addArc(0, 4)
        assert maxDistance(g, 0)[0] == 2, "Longest path of length 2"

    def testMaxDistanceWithOnlyIncomingEdges(self):
        g = self.getEmptyGraph()
        g.addVertices(3)
        g.addArc(1, 0)
        g.addArc(2, 0)
        l, d = maxDistance(g, 0)
        assert math.isinf(l)
        assert d == [0, float("inf"), float("inf")]

    def testDistanceMatrixDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        inf = float("infinity")
        assert distanceMatrix(g) == [
            [0, 1, inf, inf, inf, inf, inf, inf],
            [inf, 0, inf, inf, inf, inf, inf, inf],
            [1, 2, 0, 1, inf, inf, inf, inf],
            [inf, inf, inf, 0, inf, inf, inf, inf],
            [2, 3, 1, 2, 0, 1, 2, 3],
            [inf, inf, inf, inf, inf, 0, 1, 2],
            [inf, inf, inf, inf, inf, inf, 0, 1],
            [inf, inf, inf, inf, inf, inf, inf, 0],
        ], "That is one huge matrix"

    def testDiameterDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert math.isinf(diameter(g)), "This Directed Graph has no SCC, so diameter infinity"

    def testRadiusDirectedTreeGraph(self):
        g = self.getDirectedTreeGraph()
        assert radius(g) == 3, "The radius of this tree is 3"


class UndirectedGraphTests(GraphSetups):
    """
    This class contains specific code meant to be used
    only if testing an undirected graph
    """

    def testTopSort1EmptyGraph(self):
        g = self.getEmptyGraph()
        func = topSort1
        self.assertRaises(ValueError, func, g, 0)
        self.assertRaises(ValueError, func, g, -1)
        self.assertRaises(ValueError, func, g, 1)

    def testTopSort11V0EGraph(self):
        g = self.get1V0EGraph()
        self.assertRaises(ValueError, topSort1, g, 0), "Cannot topsort an undirected graph"

    def testTopSort11V1EGraph(self):
        g = self.get1V1EGraph()
        self.assertRaises(ValueError, topSort1, g, 0), "Cannot topsort an undirected graph"

    def testTopSort2EmptyGraph(self):
        g = self.getEmptyGraph()
        self.assertRaises(ValueError, topSort2, g), "Cannot topsort an undirected graph, even if it is empty"

    def testTopSort21V0EGraph(self):
        g = self.get1V0EGraph()
        self.assertRaises(ValueError, topSort2, g), "Cannot topsort an undirected graph"

    def testTopSort21V1EGraph(self):
        g = self.get1V1EGraph()
        self.assertRaises(ValueError, topSort2, g), "Cannot topsort an undirected graph"


class DirectedAdjListsGraphAlgorithmTests(unittest.TestCase, AlgorithmTests, DirectedGraphTests):
    def getGraphInstance(self):
        return DirectedAdjListsGraph()


class UndirectedAdjListsGraphAlgorithmTests(unittest.TestCase, AlgorithmTests, UndirectedGraphTests):
    def getGraphInstance(self):
        return UndirectedAdjListsGraph()


class DirectedAdjMatrixGraphAlgorithmTests(unittest.TestCase, AlgorithmTests, DirectedGraphTests):
    def getGraphInstance(self):
        return DirectedAdjMatrixGraph()


class UndirectedAdjMatrixGraphAlgorithmTests(unittest.TestCase, AlgorithmTests, UndirectedGraphTests):
    def getGraphInstance(self):
        return UndirectedAdjMatrixGraph()


if __name__ == "__main__":
    unittest.main()
