"""Representations for the internal state of graphs in the pyGraphADT
package

This module specifies the interface of representations which may be used within the graph package.

A Representation encapsulates the design decision of how the actual graph information is stored,
and exposes a consistent interface, regardless of whether it is a list or matrix.

Two standard representations are also included: AdjacencyLists and AdjacencyMatrix.
"""
from abc import ABCMeta, abstractmethod
from functools import reduce

__all__ = ["Representation", "AdjacencyLists", "AdjacencyMatrix"]


class Representation(metaclass=ABCMeta):
    """
    This abstract class specifies the interface for the representation of the state within a graph.

    The state of a graph is defined to be the vertices contained within it, as well as the arcs
    between those vertices.  Vertices are always numbered consecutively, starting from zero.  Deleting
    a vertex causes all higher numbered vertices to be renumbered.

    Arcs are two-ary relations upon the set of vertices, and are directed.  Arcs retain their meaning
    during deletion, i.e. they must be renumbered along with the vertices.  An arc incident or outgoing
    from a deleted vertex is also deleted.

    Edges are not explicitly modelled by this representation.  However, by convention, the user may
    choose to represent an edge by a pair of symmetric arcs together represents an edge.

    This class is meant to be used internally within the pyGraphADT package only.  If you want to
    model a graph, used a subclass of the Graph type in graphadt.graphtypes.
    """

    @abstractmethod
    def read(self, stream, defaultConstructor):
        """
        Sets the state of this representation to the data contained in some stream s.

        This method lets this representation set its state from a stream.
        The parameter defaultConstructor should be the method to call to connect two
        Vertices in this graph.

        @param stream: the stream must support the 'readlines()' method.
        @param defaultConstructor: a method which can be called upon this object
            which specifies how connections are to be made.
        @type defaultConstructor: unbound function
        @type stream: filestream supporting readlines()
        """

        raise NotImplementedError()

    @abstractmethod
    def empty(self):
        """
        Removes all vertices from this graph.
        """
        raise NotImplementedError()

    @abstractmethod
    def addVertices(self, n):
        """
        Creates records for n new vertices

        @param n: label of a vertex in this representation
        @type n: number
        """
        raise NotImplementedError()

    @abstractmethod
    def removeVertex(self, i):
        """
        This method deletes the record of vertex labelled i, and all arcs connected to it.
        All the vertices labelled higher than this vertex are relabelled.

        @param i: a label of a vertex in this representation
        @type i: number

        @raises ValueError: if no such label i exists
        """
        raise NotImplementedError()

    @abstractmethod
    def addArc(self, i, j):
        """
        Records that in this state there is an arc from vertex with label i to vertex with label
        j

        @param i: a label of a vertex in this representation
        @type i: number
        @param j: a label of a vertex in this representation
        @type j: number

        @raises ValueError: if no such labels exist

        """

        raise NotImplementedError()

    @abstractmethod
    def removeArc(self, i, j):
        """
        Removes a directional connection from vertex i to vertex j, from the state of this
        representation if it exists.  If such a connection does not exist, no error is raised.

        @param i: a label of a vertex in this representation
        @type i: number
        @param j: a label of a vertex in this representation
        @type j: number

        @raises ValueError: if no such labels exist
        """
        raise NotImplementedError()

    @abstractmethod
    def isArc(self, i, j):
        """
        Returns True if and only if there is an arc connecting i to j.

        @param i: a label of a vertex in this representation
        @type i: number
        @param j: a label of a vertex in this representation
        @type j: number

        @returns:   True if there is an arc from vertex with label i to vertex with
            label j in this state.
            False otherwise
        @rtype: boolean

        @raises ValueError: if no such labels exist
        """

        raise NotImplementedError()

    @abstractmethod
    def inDegree(self, i):
        """
        Returns the number of other vertices which are connected to i

        @param i: a label of a vertex in this representation
        @type i: number

        @return:    the number of arcs incident on i, as integer.
        @rtype: number
        @raises ValueError: if no such labels exist
        """

        raise NotImplementedError()

    @abstractmethod
    def degree(self, i):
        """
        Returns the number of vertices to which vertex i connects

        @param i: a label of a vertex in this representation
        @type i: number
        @returns:    the number of arcs outgoing from i, as integer.
        @rtype: number
        Raises:     ValueError if no such label i exists.
        """

        raise NotImplementedError()

    @abstractmethod
    def neighbours(self, i):
        """
        Returns an iterable collection of all the vertices to which
        the vertex with index i connects

        @param i: a label of a vertex in this representation
        @type i: number

        @returns: all the vertices j, such that there is an arc (i, j).  J must
            be sorted, and safe for modification.
        @rtype: list

        @raises ValueError: if no such labels exist
        """

        raise NotImplementedError()

    @abstractmethod
    def size(self):
        """
        Returns the number of arcs in this graph, as an integer.

        @return: the number of arcs in this representation
        @rtype: number
        """

        raise NotImplementedError()

    def selfEdges(self):
        """
        Returns the number of arcs in this representation from any vertex to itself, as an integer.

        @return: the number of self loops in this representation
        @rtype: number
        """

        selfEdges = 0
        for i in range(self.order()):
            if i in self.neighbours(i):
                selfEdges += 1
        return selfEdges

    @abstractmethod
    def order(self):
        """
        Returns the number of vertices in this graph.

        @return: the number of vertices in this graph
        @rtype: number
        """

        raise NotImplementedError()

    @abstractmethod
    def __str__(self):
        """
        Returns a string representation of this object's state.  It should always be the case that
        self.read() and str(self) should be compatible: the output of str should be valid input
        for self.read()

        @rtype: string
        @returns: a string representation of this object's state
        """
        raise NotImplementedError()


class AdjacencyLists(Representation):
    """
    AdjacencyLists extends and realises the abstract class Representation.

    AdjacencyLists stores the state of vertices and arcs as a list of lists.
    The arc (i,j) is represented by the list at element i containing the value j.
    """

    def __init__(self):
        self._adj = []  # Creates an internal empty array signifying an empty graph

    #   Mutators
    def empty(self):
        self._adj = []

    def addVertices(self, n):
        if not 0 <= n:
            raise ValueError("Argument cannot be negative: n" % (n))

        for i in range(n):
            self._adj.append([])

    def removeVertex(self, i):
        if not 0 <= i < self.order():
            raise ValueError(f"Arguments out of bounds: v = {i}")

        del self._adj[i]  # Delete this vertex's adjacency list

        for otherVertex in range(self.order()):
            current = self._adj[otherVertex]

            try:
                current.remove(i)  # Remove vertex i if it appears
            except ValueError:
                # It didn't appear
                pass

            for j in range(len(current)):  # Relabel all higher vertices
                if current[j] > i:
                    current[j] -= 1

    def addArc(self, i, j):
        if not (0 <= i < self.order() and 0 <= j < self.order()):
            raise ValueError("Arguments out of bounds: i = %s and j = %s" % (i, j))

        if not self.isArc(i, j):  # This could be faster if instead of adjacency lists
            self._adj[i].append(j)  # We used adjacency sets.

    def removeArc(self, i, j):
        if not (0 <= i < self.order() and 0 <= j < self.order()):
            raise ValueError("Arguments out of bounds: i = %s and j = %s" % (i, j))

        try:
            self._adj[i].remove(j)
        except ValueError:
            pass  # Not Found

    def isArc(self, i, j):
        if not (0 <= i < self.order() and 0 <= j < self.order()):
            raise ValueError("Arguments out of bounds: i = %s and j = %s" % (i, j))

        return j in self._adj[i]

    def degree(self, i):
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        return len(self._adj[i])

    def inDegree(self, i):
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        retval = 0
        for j in range(self.order()):
            if self.isArc(j, i):
                retval += 1
        return retval

    def neighbours(self, i):
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        return sorted(self._adj[i])  # Returning a copy that is safe for modification

    def order(self):
        return len(self._adj)

    def size(self):
        return reduce(lambda x, y: x + y, map(len, self._adj), 0)  # Add up  # The lengths of every row  # Initial value 0

    def __str__(self):
        retval = ""
        retval += str(self.order()) + "\n"
        for i in range(self.order()):
            retval += " ".join(map(str, self.neighbours(i))) + "\n"
        return retval

    def read(self, stream, defaultConnector):
        self.empty()
        order = int(stream.readline().strip())
        self.addVertices(order)
        for i in range(order):
            for neighbour in map(int, stream.readline().strip().split()):
                defaultConnector(i, neighbour)


class AdjacencyMatrix(Representation):
    """
    AdjacencyMatrix extends and realises the abstract class Representation.

    AdjacencyMatrix stores the state of vertices and arcs as a square matrix.
    The arc (i,j) is represented by a value of True at element (i, j).

    Note : In python, multi dimensional arrays are not built in, so this
    matrix is stored as a jagged array, or an array of arrays.
    """

    def __init__(self):
        self._adj = []

    def empty(self):
        self._adj = []

    def addVertices(self, n):
        if not 0 <= n:
            raise ValueError("Argument cannot be negative: n" % (n))

        old_order = len(self._adj)

        for r in self._adj:
            r += [0] * n  # Extend each existing row

        for i in range(n):
            self._adj.append([0] * (old_order + n))  # Add n new rows

    def removeVertex(self, i):
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        del self._adj[i]
        for row in self._adj:
            del row[i]

    def addArc(self, i, j):
        if not (0 <= i < self.order() and 0 <= j < self.order()):
            raise ValueError("Arguments out of bounds: i = %s and j = %s" % (i, j))

        self._adj[i][j] = True

    def removeArc(self, i, j):
        if not (0 <= i < self.order() and 0 <= j < self.order()):
            raise ValueError("Arguments out of bounds: i = %s and j = %s" % (i, j))

        self._adj[i][j] = False

    def isArc(self, i, j):
        if not (0 <= i < self.order() and 0 <= j < self.order()):
            raise ValueError("Arguments out of bounds: i = %s and j = %s" % (i, j))

        return self._adj[i][j]

    def degree(self, i):  # Count the Trues in a row
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        return reduce(lambda x, y: x + y, self._adj[i], 0)  # Add up all the booleans  # True == 1  # Starting with 0

    def inDegree(self, i):  # Count the Trues in a column
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        return reduce(lambda x, y: x + y[i], self._adj, 0)  # Add up the ith element  # In every row  # initial value of 0

    def neighbours(self, i):
        if not 0 <= i < self.order():
            raise ValueError("Argument out of bounds: i = %s" % i)

        neighbours = []
        for j in range(self.order()):
            if self._adj[i][j]:
                neighbours.append(j)
        return neighbours

    def order(self):
        return len(self._adj)

    def size(self):
        add = lambda x, y: x + sum(y)
        return reduce(add, self._adj, 0)

    def __str__(self):
        retval = ""
        n = self.order()
        retval += str(n) + "\n"
        for i in range(n):
            for j in range(n):
                if self._adj[i][j]:
                    retval += "1"
                else:
                    retval += "0"
                if not j == n - 1:
                    retval += " "
            retval += "\n"
        return retval

    def read(self, stream, defaultConnector):
        self.empty()
        order = int(stream.readline().strip())
        self.addVertices(order)
        for i in range(order):
            for neighbour, flag in enumerate(map(int, stream.readline().strip().split())):
                if flag == 1:
                    defaultConnector(i, neighbour)
