"""Graph abstract data types of the pyGraphADT package.

Python version by Danver Braganza
designed to follow the signatures of code originally authored by
Michael J. Dinneen, with contributions by Sonny Datt.

2010 February

This module is part of the pyGraphADT package, and contains the definitions of
graph types.

This module is designed for use in the teaching of the Computer Science 220
course at the University of Auckland.
"""

from graphadt import representations

from abc import ABCMeta, abstractmethod
from collections import defaultdict

__all__ = [
    "Graph",
    "DirectedGraph",
    "UndirectedGraph",
    "DirectedAdjListsGraph",
    "DirectedAdjMatrixGraph",
    "UndirectedAdjListsGraph",
    "UndirectedAdjMatrixGraph",
    "WeightedDirectedAdjListsGraph",
    "WeightedUndirectedAdjListsGraph",
]


class Graph(object, metaclass=ABCMeta):
    """This is the basic definition of a graph. A graph as defined by this class
    is to be a collection of consecutively-numbered vertices starting from zero,
    and a collection of 2-ary relations between these vertices, representing
    arcs or edges.

    Graphs are implemented as wrappers of encapsulated Representation objects.
    This means that graph object code for Adjacency Lists and Adjacency Matrix
    representations can be identical.

    Constructors:
    -__init__(self, representation) -> default constructor
    -copy(cls, g) -> copy constructor
    -read(cls, stream) -> file reader constructor

    Mutators:
    -addVertices(n)
    -addEdge(i, j)
    -removeVertex(i)
    -removeEdge(i, j)

    Accessors:
    -isEdge(i, j)
    -degree(i)
    -inDegree(i)
    -order()
    -size()
    -neighbours(i)

    Utility:
    -__repr__() -- print a string representation of this graph

    28/01/10: Graph is now an abstract class.  This will not work in python 2.5

    """

    def __init__(self, representation):
        """Constructs an instance of this graph, and sets the hidden representation
        to the parameter.

        @param representation: the state of the graph to be copied
        @type representation: graphADT.representations.Representation
        """
        self._representation = representation

    @classmethod
    def copy(cls, g):
        """Creates an instance of cls, which is a copy of g.

        The instance returned by this method is distinct from g, is of type
        cls, and is at least as connected as g.

        g need not be of the same type as cls.  In fact, copying a
        directed graph to an undirected graph is a standard way to create
        its underlying graph.

        @param cls: a concrete subclass of graph
        @type cls: class
        @param g: the graph to be copied
        @type g: graphadt.graphtypes.Graph

        Warning:    A Graph itself is an abstract class, Graph.copy(g) is an
        invalid method call.
        """

        newinstance = cls()
        n = g.order()
        newinstance.addVertices(n)
        for i in range(n):
            for j in g.neighbours(i):
                newinstance.defaultConnector(i, j)
        return newinstance

    @abstractmethod
    def defaultConnector(self, i, j):
        """Joins i and j using the default connector type for this graph.

        The default connector is the connector (Arc or Edge) used when constructing this graph
        from other representations, such as when reading from file streams or copying
        other graphs

        This method is not intended for use in external code, it is for internal code which
        does not know whether to create arcs or edges.

        @param i: The label of the vertex from which the connection will be made
        @type i: integer
        @param j: The label of the vertex to which the connection will be made
        (order is only important if the defaultConnector is an arc)
        @type j: integer
        """
        return False

    def addVertices(self, n):
        """Adds n vertices to this graph.

        This method adds n unconnected vertices to this graph.  The vertices are numbered
        consecutively, continuing from the highest numbered vertex already in the graph.

        @param n: the number of vertices to add
        @type n: integer
        @raise ValueError: if n is negative
        """
        self._representation.addVertices(n)

    def removeVertex(self, i):
        """Removes vertex with label i from this graph.

        This method removes the vertex with label i from this graph. All edges connected to it
        are also removed.  All the vertices labelled higher than this vertex are relabelled.

        @param i: the label of the vertex to remove
        @type i: integer
        @raise ValueError: if i is not a valid vertex label in this graph.
        """
        self._representation.removeVertex(i)

    def addEdge(self, i, j):
        """        Creates an edge between i and j

        This method creates an edge between i and j.  Edges are bi-directional, so the order of
        the parameters i and j does not matter.  Self-loops, i.e. edges where i == j, are allowed.

        @param i: The label of the vertex from which the connection will be made
        @type i: integer
        @param j: The label of the vertex to which the connection will be made
        @type j: integer
        @raise ValueError: if either of i or j is not a valid vertex label in this graph.

        """
        self._representation.addArc(i, j)
        self._representation.addArc(j, i)

    def removeEdge(self, i, j):
        """        Removes any edge or arcs between i and j

        This method removes the edge between i and j.  If such an edge does not exist, this
        method will remove the arc (i, j) or (j, i), if they exist.

        @param i: The label of a vertex in this graph
        @type i: integer
        @param j: The label of a vertex in this graph
        @type j: integer
        @raise ValueError: if either of i or j is not a valid vertex label in this graph.
        """
        self._representation.removeArc(i, j)
        self._representation.removeArc(j, i)

    def isEdge(self, i, j):
        """        Returns True if and only if there exists and edge between i and j

        @param i: The label of a vertex in this graph
        @type i: integer
        @param j: The label of a vertex in this graph
        @type j: integer
        @raise ValueError: if either of i or j is not a valid vertex label in this graph.
        @return: True iff there is a biderectional edge between i and j
        @rtype: boolean
        """
        return self._representation.isArc(i, j) and self._representation.isArc(j, i)

    def degree(self, i):
        """        Returns the outdegree of vertex i.

        This method returns the outdegree of vertex i, which is the number of vertices
        directly connected by outgoing edges or arcs from i.

        @param i: The label of a vertex in this graph
        @type i: integer

        @raise ValueError: if i is not a valid vertex label in this graph.
        @return: the outdegree of vertex with label i.
        @rtype: number
        """
        return self._representation.degree(i)

    @abstractmethod
    def inDegree(self, i):
        """Returns the indegree of vertex i.

        This method returns the indegree of vertex i, which is the number
        of vertices which directly connect to i with incoming edges or arcs.

        @param i: The label of a vertex in this graph
        @type i: integer

        @raise ValueError: if i is not a valid vertex label in this graph.
        @return: the indegree of vertex with label i.
        @rtype: number

        """
        raise NotImplementedError()

    def order(self):
        """Returns the number of vertices within this graph.

        @return: the number of vertices in this graph.
        @rtype: number

        """
        return self._representation.order()

    @abstractmethod
    def size(self):
        """Returns the number of edges of an undirected graph or the number of
        arcs in an directed graph.

        @return: the number of connectors in this graph
        @rtype: number

        """
        raise NotImplementedError()

    def neighbours(self, i):
        """Returns an iterable collection of all the vertices to which the vertex
        with index i connects

        This method returns an iterable collection of all the neighbours of vertex i.
        A vertex j is defined to be the neighbour of vertex i if there exists an edge
        between i and j, or if there exists an arc from i to j.


        @param i: The label of a vertex in this graph
        @type i: integer

        @raise ValueError: if i is not a valid vertex label in this graph.
        @return: the labels of the neighbours of vertex i
        @rtype: list
        """
        return self._representation.neighbours(i)

    neighbors = neighbours  # Permit the American spelling

    def __repr__(self):
        """        Equivalent of Java's toString

        This method returns a string representation of this graph.  The precise format
        of this representation depends on the type of representation encapsulated
        within this graph.

        @return: string representation of this graph
        @rtype: string

        """
        return str(self._representation)

    @classmethod
    def read(cls, stream):
        """        Constructs a graph to be equivalent to some stream data.

        This method reads a stream of data from a file or similar source, and creates an
        instance of type concrete type cls, which is set up to be equivalent to the data.

        Note that an instance of type cls must accept the format of data being read.

        @param cls: a concrete subclass of graph
        @type cls: class
        @param stream: the data to be duplicated
        @raise ValueError: if stream is of the wrong format

        """
        newinstance = cls()
        newinstance._representation.read(stream, newinstance.defaultConnector)
        return newinstance


class DirectedGraph(Graph):
    """    This class defines a Directed Graph.  The Directed Graph class extends the Graph
    class and enables the creation of directed arcs.

    This class provides several methods for modifying and accessing arcs, as well as extending
    certain accessors to provide more accurate functionality.

    Mutators:
    -addArc(n)
    -removeArc(i, j)

    Accessors:
    -isArc(i, j)
    -inDegree(i)
    -size()

    If a Directed Graph contains two symmetric arcs, it is considered to contain the edge.
    """
    def addArc(self, i, j):
        """        Creates an arc from i to j

        This method creates an arc from i to j.  Arcs are uni-directional, and so the order of
        the parameters i and j does matter.  Self-loops, i.e. arcs where i == j, are allowed,
        but are indistinguishable from edges.


        @param i: The label of a vertex in this graph
        @type i: integer
        @param j: The label of a vertex in this graph
        @type j: integer
        @raises ValueError: if either i or j is not a valid vertex label in this graph.
        """
        self._representation.addArc(i, j)

    def removeArc(self, i, j):
        """Removes an arc from i to j

        This method removes an arc from i to j.  Note that if an edge between i and j
        exists, this method can 'split' the edge and remove the outbound arc, so that
        the arc (j, i) in the other direction, is left.



        """
        self._representation.removeArc(i, j)

    def isArc(self, i, j):
        """Returns True if and only if j is adjacent to i.

        This method returns true if there is an arc from i to j.  However, it also returns
        true if there is an edge between i and j.

        @param i: The label of a vertex in this graph
        @type i: integer
        @param j: The label of a vertex in this graph
        @type j: integer
        @raises ValueError: if either i or j is not a valid vertex label in this graph.

        @return: True iff there is an arc between i and j
        @rtype: boolean

        """
        return self._representation.isArc(i, j)

    def inDegree(self, i):
        """Returns the number of arcs and edges going into i.

        Parameter: i -- a non-negative integer representing a vertex in this graph

        Raises : ValueError if either i or j is not a valid vertex label in this graph.
        """
        return self._representation.inDegree(i)

    def size(self):
        """Returns the number of arcs in this graph.

        This method returns the number of arcs in this graph.  An edge counts as two
        arcs.

        @return: the number of arcs in this graph
        @rtype: integer

        """
        return self._representation.size()

    def defaultConnector(self, i, j):
        """        The default connector of this graph is the arc.

        """
        self.addArc(i, j)


class UndirectedGraph(Graph):
    """    This class defines the Undirected Graph.  The Undirected Graph class extends the Graph
    class, and has some behaviour customised for an undirected graph.
    """
    #    def addArc(self, i, j):
    #    Undefined for Undirected Graphs

    #    def removeArc(self, i, j):
    #    Undefined for Undirected Graphs

    #    def isArc(self, i, j):
    #    Undefined for Undirected Graphs

    def defaultConnector(self, i, j):
        """        The default connector of an undirected graph is an edge.
        """
        self.addEdge(i, j)

    def inDegree(self, i):
        """Returns the degree of vertex i, since indegree and outdegree are
        equivalent in a directed graph.
        @rtype: integer
        @return: the degree of vertex i
        """        # In-degree and out-degree are equivalent in undirected graph
        # But out-degree is potentially computationally cheaper
        return self.degree(i)

    def size(self):
        """Returns the number of edges of this graph.

        The size of an undirected graph is defined as the number of edges.
        Since the inner representation object only stores the number of arcs,
        this number of arcs must be halved.  However, self-arcs are NOT counted twice,
        and halving this number leads to errors.  Therefore, the number of edges
        is equal to half the number of non-self arcs, plus the self-arc.

        @return: the number of edges
        @rtype: integer
        """
        size = self._representation.size()
        selfEdges = self._representation.selfEdges()
        size -= selfEdges
        return size // 2 + selfEdges  # Divide non-self edges by two to account
        # for two arcs to an edge


class DirectedAdjListsGraph(DirectedGraph):
    """    Convenience class which constructs a directed graph and sets the inner representation
    to an AdjacencyListsRepresentation.
    """
    def __init__(self):
        DirectedGraph.__init__(self, representations.AdjacencyLists())


class DirectedAdjMatrixGraph(DirectedGraph):
    """    Convenience class which constructs a directed graph and sets the inner representation
    to an AdjacencyMatrixRepresentation.
    """
    def __init__(self):
        DirectedGraph.__init__(self, representations.AdjacencyMatrix())


class UndirectedAdjListsGraph(UndirectedGraph):
    """    Convenience class which constructs an undirected graph and sets the inner representation
    to an AdjacencyListsRepresentation.
    """
    def __init__(self):
        UndirectedGraph.__init__(self, representations.AdjacencyLists())


class UndirectedAdjMatrixGraph(UndirectedGraph):
    """Convenience class which constructs an undirected graph and sets the inner
    representation to an AdjacencyMatrixRepresentation.
    """
    def __init__(self):
        UndirectedGraph.__init__(self, representations.AdjacencyMatrix())


class WeightedGraph(Graph):
    """This abstract class manages the weights of edges of a graph.

    A weighted graph wraps an inner representation, and provides a few more functions
    for manipulating weights.  WeightedGraph is only a mixin class.

    One of the key invariants is that if this class returns True for self.isArc(i, j),
    then self.getArcWeight(i, j) must return a legitimate weight, or zero if none was set.

    However, the output of self.getArcWeight(i, j), called with arbitrary arguments,
    is not guaranteed.

    Note: a weighted graph is an abstract class, and intended to be inherited from.
    """
    def __init__(self):
        """        Creates a mapping dictionary of edge weights, and assigns all edges a
        default weight of 0

        Note: You MUST call the other Graph class's constructor _before_ calling this one.
        """
        self._weights = defaultdict(lambda: 0)

        # Warning: this is a really bad way to register THIS object to be
        # Notified whenever a arc is removed from the representation.
        # Basically, we replace the method call on the representation with an intercepting
        # call on this object, which then calls the representation method correctly.

        self._repremoveArc = self._representation.removeArc
        self._representation.removeArc = self._removeArc

    @classmethod
    def copy(cls, g):
        """Extends Graph.copy to copy weighted graphs.

        This method creates a copy graph g.  The copy is of type cls.  If g is a
        weighted graph, the weights are copied as well.
        """
        newinstance = Graph.copy(cls, g)
        if hasattr(g, "getArcWeight"):
            for i in range(g.order()):
                for j in range(g.order()):
                    newinstance.setArcWeight(i, j, g.getArcWeight(i, j))

    def setArcWeight(self, i, j, w):
        """        Sets the weight of arc (i, j) to w.

        @param i: The label of a vertex in this graph
        @type i: integer
        @param j: The label of a vertex in this graph
        @type j: integer
        @param w: the weight of arc i,j
        @raise ValueError: if i or j are not valid vertices in this graph
        @raise ValueError: if arc (i, j) does not exist
        """
        if not self._representation.isArc(i, j):
            raise ValueError("No such Arc exists")
        self._weights[i, j] = w

    def getArcWeight(self, i, j):
        """Returns the weight of arc (i, j), or zero if no weight was set.

        @param i: The label of a vertex in this graph
        @type i: integer
        @param j: The label of a vertex in this graph
        @type j: integer
        @returns: the weight of arc i, j
        @raise ValueError: if i or j are not valid vertices in this graph
        @raise ValueError: if arc (i, j) does not exist
        """
        if not self._representation.isArc(i, j):
            raise ValueError("No such Arc exists")
        return self._weights[i, j]

    def getNeighbourWeights(self, i):
        """Returns a list of weights of the neighbours of i.

        This method returns a list of the weights of the arcs out of i, in
        a direct correspondence to the list returned by neighbours(i).  In other words,
        for every element at index j in neighbours, the weight of the arc from i to
        j is the jth element of getNeighbourWeights,


        @param i: The label of a vertex in this graph
        @type i: integer
        @returns: the weights of all neighbours of i
        @rtype: list
        @raise ValueError: if i is not a valid vertex in this graph

        """
        return [self.getArcWeight(i, j) for j in self._representation.neighbours(i)]

    def _removeArc(self, i, j):
        """Removes a given arc from this graph, and cleans up the weights.

        @param i: vertex from which the arc leads
        @type i: integer
        @param j: vertex from which the arc leads
        @type j: integer
        """
        self._repremoveArc(i, j)  # This is really bad programming practice.  This is a bound method
        # Of a different object, completely breaking encapsulation
        # However, the solution would probably require more work, or some
        # kind of Aspect Oriented Programming, which I don't have time to
        # implement

        if (i, j) in self._weights:
            del self._weights[i, j]

    def removeVertex(self, v):
        """Removes a given vertex from this graph, and cleans up the weights.

        @param v: label of a vertex
        @type v: integer

        @raises ValueError: if v does not represent a vertex in this graph.
        """
        Graph.removeVertex(self, v)
        # NOTE: This copies the entire graph to avoid modifying the dictionary
        # in-place. A more efficient algorithm is possible, but since n is
        # usually small, this isn't relevant.
        for (i, j), weight in list(self._weights.items()):
            if i < v and j < v:
                continue
            del self._weights[i, j]
            if i == v or j == v:
                continue
            if i > v:
                i -= 1
            if j > v:
                j -= 1
            self._weights[i, j] = weight


class WeightedDirectedAdjListsGraph(WeightedGraph, DirectedAdjListsGraph):
    """Convenience class which uses multiple inheritance to achieve weighted, Directed
    behaviour using an underlying Adjacency Lists Representation.

    Note: this class has not been completely tested.  Some faults may still exist.
    """
    def __init__(self):
        DirectedAdjListsGraph.__init__(self)
        WeightedGraph.__init__(self)


class WeightedUndirectedAdjListsGraph(WeightedGraph, UndirectedAdjListsGraph):
    """Convenience class which uses multiple inheritance to achieve weighted, Directed
    behaviour using an underlying Adjacency Lists Representation.

    Note: this class has not been completely tested.  Some faults may still exist.
    """
    def __init__(self):
        UndirectedAdjListsGraph.__init__(self)
        WeightedGraph.__init__(self)
