"""Unweighted graph algorithms of the pyGraphADT package.

This module is part of the pyGraphADT package, and contains a number of basic
graph algorithms which operate upon the graph abstract types included in this
same package.

This module is designed to be used in the teaching of the Computer Science 220
course at the University of Auckland.
"""
from math import isinf
from collections import deque

from graphadt.graphtypes import DirectedAdjListsGraph, UndirectedAdjListsGraph
from graphadt.graphtypes import UndirectedGraph, DirectedGraph


__all__ = [
    "getBFSParents",
    "BFS",
    "DFS",
    "isConnected",
    "isStronglyConnected",
    "strongComponents",
    "getDFSParents",
    "isAcyclic",
    "girth",
    "isBipartite",
    "topSort1",
    "topSort2",
    "maxDistance",
    "distanceMatrix",
    "diameter",
    "radius",
]


def getBFSParents(g, v):
    """Returns a list containing the direct ancestors of every vertex as
    discovered during a breadth-first search starting from node v.

    This function performs a breadth-first search starting at the vertex
    labelled v.  A list is returned such that the element at index i
    is the label of the direct parent of vertex i.  If a vertex has not
    been discovered during the BFS traversal (due to being disconnected)
    its parent is set to None.  Furthermore, the parent of v is set to
    itself.

    Note: The correct labels of the parents are used.  This may conflict with
    some versions of the Java implementation, where indices are
    incremented by one.

    @type  g: L{graphadt.graphtypes.Graph}
    @param g: a graph
    @type  v: number
    @param v: the integer label of a vertex

    @returns: index of parent or None
    @rtype: list of integers


    @raise ValueError: if v is not the label of a vertex in g
    """

    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")

    parents = [None] * g.order()
    parents[v] = v  # If v is our root, we say it is its own parent

    for vfrom, vto in BFSIt(g, v):
        if parents[vto] is None:
            parents[vto] = vfrom
    return parents


def BFS(g, v):
    """Performs breadth-first search from vertex v and returns number of vertices
    discovered and a list showing the order vertices was discovered.

    This function performs a breadth-first search starting from vertex v, and
    returns a tuple containing two values.  The first value is the number of
    vertices discovered during breadth-first search, and the second a list where
    element i is the order in which vertex i was discovered.  If vertex i is not
    discovered by this search, then element i of this list is set to None.

    Note: Numberings in this list start from zero.

    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}
    @param v: the integer label of a vertex
    @type v: number

    @return: an integer containing the number of vertices dicovered,
        a list comprised of integers and one or more None.
    @rtype: tuple

    @raise ValueError: if v is not the label of a vertex in g
    """

    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")
    levelorder = [None] * g.order()
    level = 0
    levelorder[v] = level

    for vfrom, vto in BFSIt(g, v):
        if levelorder[vto] == None:
            level += 1
            levelorder[vto] = level

    level += 1
    return level, levelorder


def BFSIt(g, v):
    """Iterates over the vertices in graph g in breadth-first order.

    This function iterates over the vertices in a graph is breadth-first order.
    It will repeat vertices if there are cycles in the graph, but will not
    expand them.  This is so that code which makes use of this iterator can
    detect cycles, while trusting that this iterator will never loop infinitely.

    This function is not intended to be called by external code.

    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}
    @param v: the integer label of a vertex
    @type v: number
    @return:   yields (vfrom, vto), where vfrom is the current parent vertex,
        and vto is the vertex being discovered.
    @rtype: tuple

    @raises ValueError: if v is not the label of a vertex in g
    @raises StopIteration: if all vertices have been discovered.
    """
    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")
    seen = [False] * g.order()

    seen[v] = True
    toGrow = deque([v])

    while toGrow:
        vfrom = toGrow.popleft()
        for vto in g.neighbours(vfrom):
            yield vfrom, vto  # Whether we've seen it or not, yield it
            if not seen[vto]:
                seen[vto] = True
                toGrow.append(vto)  # Only if we've not seen it, queue it


class Countpair(object):
    def __getattribute__(self, name):
        object.__setattr__(self, name, object.__getattribute__(self, name) + 1)
        return object.__getattribute__(self, name)

    def __init__(self):
        self.pre = 0
        self.post = 0


def DFS(g, v):
    """Performs depth-first search from vertex v, and returns number of vertices
    discovered as well as two lists showing the pre-order and post-order in which
    vertices were discovered.

    This function performs a depth-first search starting from vertex v, and
    returns a tuple containing three values.  The first value is the number of
    vertices discovered during the search. The second value is a list of
    preorders, where element i is the order in which vertex i was first visited.
    The third value is a list of postorders, where element i is the order it
    which vertex i was lat visited.  If a node is never visited, its preorder
    and postorder will both be None.

    Note: Numbering of order in both lists start from zero.

    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}
    @param v: the integer label of a vertex
    @type v: number

    @return: a tuple of: integer representing total vertices discovered
        preorder : a list containing the preorders of every vertex
        postorder: a list containing the postorders of every vertex
    @rtype: tuple

    @raise ValueError: if v is not the label of a vertex in g
    """

    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")

    def doDFS(g, v, preorder, postorder, count):
        """
        """
        preorder[v] = count.pre
        for i in g.neighbours(v):
            if preorder[i] is None:
                preorder, postorder, count = doDFS(g, i, preorder, postorder, count)
        postorder[v] = count.post
        return preorder, postorder, count

    preorder = [None] * g.order()
    postorder = [None] * g.order()
    preorder, postorder, count = doDFS(g, v, preorder, postorder, Countpair())
    return postorder[v], preorder, postorder


def isConnected(g):
    """True if and only if the graph g is connected.

    This function checks for connectedness of graph g by converting it into an
    undirected graph, and then performing breadth-first search.  G is considered
    connected if this breadth-first search is able to discover all nodes.

    Note: if g is a directed graph, then this function returns True if g is weakly
    connected.

    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}

    @return: True if and only if graph g is connected
        False otherwise
    @rtype: boolean
    """

    if g.order() == 0:
        return True  # Trivially connected

    g = UndirectedAdjListsGraph.copy(g)
    return BFS(g, 0)[0] == g.order()


def isStronglyConnected(g):
    """Returns True if and only if the graph g is strongly connected.

    This function returns true if and only if g is strongly connected.  A
    strongly connected graph is defined as one where for every pair of vertices
    u and v, there exists a path from u to v.

    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}

    @return: True if and only if graph g is strongly connected
        False otherwise
    @rtype: boolean

    """

    n = g.order()
    if g.order() == 0:
        return True

    if DFS(g, 0)[0] < n:
        return False

    reversed = DirectedAdjListsGraph()
    reversed.addVertices(n)
    for i in range(n):
        for j in g.neighbours(i):
            reversed.addArc(j, i)

    return DFS(reversed, 0)[0] == n


def strongComponents(g):
    """Finds the strong components of graph g.

    This function finds the strong components of graph g.  For any pair of
    vertices v, u, if there is a path from v to u and a path from u to v, then
    u and v must belong to the same strongly-connected-component.

    This function returns a list where the element at index i is the label of
    the component to which vertex i belongs.  If graph g is strongly connected,
    every element in this list will be 0, since every vertex is in component 0.

    Note: Numberings in this list start from zero.
    To find the number of components, use max(strongComponents(g)) + 1


    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}

    @return:    a list comprised of integers.

    @raises ValueError: if v is not the label of a vertex in g
    """

    scc = [None] * g.order()
    component = 0

    for i in range(g.order()):
        if scc[i] is None:
            scc[i] = component
            trash, comp = BFS(g, i)
            for j, depth in enumerate(comp):
                if depth is not None:
                    if BFS(g, j)[1][i] is not None:
                        scc[j] = component
            component += 1
    return scc


def getDFSParents(g, v, parents=None):
    """Returns a list containing the direct ancestors of every vertex as
    discovered during a depth-first search starting from node v.

    This function performs a depth-first search starting at the vertex
    labelled v.  A list is returned such that the element at index i
    is the label of the direct parent of vertex i.  If a vertex has not
    been discovered during the DFS traversal (due to being disconnected)
    its parent is set to None.  Furthermore, the parent of v is set to
    itself.

    @param g: a graph
    @type g: L{graphadt.graphtypes.Graph}
    @param v: the integer label of a vertex
    @type v: number


    @return:    a list comprised of integers and one or more None.
    @rtype: list

    @Raises ValueError: if v is not the label of a vertex in g

    Note: The correct labels of the parents are used.  This may conflict with
    some versions of the Java implementation, where indices returned are
    incremented by one.
    """

    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")

    if parents is None:
        parents = [None] * g.order()
    if parents[v] == None:
        parents[v] = v

    for i in g.neighbours(v):
        if parents[i] is None:
            parents[i] = v
            getDFSParents(g, i, parents)

    return parents


def isAcyclic(g):
    """Returns True if and only if g is an Acyclic graph.

    This function checks g for cycles.  If g is an undirected graph, cycles of
    length 2 will be ignored.  If g is a directed graph, then any edges in g will
    count as a cycle of length 2.

    Any self-loops in g will also be counted as cycles.

    @param g: the graph
    @type g: graphadt.graphtypes.Graph

    @rtype: boolean
    @return:True if and only if g is acyclic
    """
    n = g.order()
    if g.order() == 0:
        return True

    if isinstance(g, UndirectedGraph):
        return isAcyclicUndirected(g)

    span = [False] * n

    count = 0
    for i in range(n):
        if span[i]:
            continue

        componentcount, preorder, postorder = DFS(g, i)
        for j in range(n):
            if preorder[j] is not None:
                span[j] == True

            for k in g.neighbours(j):
                if k == j:
                    return False

                if not preorder[j]:
                    continue

                if preorder[k] < preorder[j]:
                    return False

            count += componentcount
            if count == n:
                break  # All vertices spanned
    return True


def isAcyclicUndirected(g):

    """True if, given that g is an undirected graph, it is also acyclic.

    Not meant to be called by external code

    @param g: the graph
    @type g: graphadt.graphtypes.Graph


    @returns:True if and only if g is acyclic
    @rtype: boolean


    """
    n = g.order()
    m = g.size()

    if m > n:
        return False

    components = max(strongComponents(g)) + 1

    return n == m + components


def girth(g):
    """Returns the girth of g.

    This function only returns values for graphs with girth >= 3 or 1.
    Girth 1 is caused by self-loops.
    Girth >= 3 is the normal case of cycles
    Girth infinity indicates no cycles
    Girth 2 is never returned, because every undirected edge would otherwise
    create a cycle of girth 2.  If the only cycle which exists is of girth 2,
    infinity is returned instead.

    @param g: the graph
    @type g: graphadt.graphtypes.Graph

    @returns:  length of smallest cycle in g
    @rtype: integer or infinity


    """

    best = float("infinity")
    n = g.order()
    for i in range(n):
        depth = [None] * n
        depth[i] = 0
        parent = [None] * n
        for vfrom, vto in BFSIt(g, i):
            if depth[vto] == None:
                depth[vto] = depth[vfrom] + 1
                parent[vto] = vfrom
            else:
                if not parent[vfrom] == vto:
                    looplength = depth[vfrom] + depth[vto] + 1
                    best = min(best, looplength)
    return best


def isBipartite(g):

    """Returns True if and only if g is bipartite

    This function verifies whether g is bipartite by attempting a 2-colouring
    of the vertices.  If the colouring succeeds, the function returns True, else
    False

    @param g: the graph
    @type g: graphadt.graphtypes.Graph

    @rtype: boolean
    @returns:  True if and only if g is bipartite

    """

    g = UndirectedAdjListsGraph.copy(g)  # important for when arbitrary choices can conflict

    n = g.order()
    colour = [None] * n
    for v in range(n):
        if not colour[v] == None:  # if we've seen it before
            continue  # never mind about it
        colour[v] = False  # Arbitrary choice for this one
        toGrow = deque([v])
        while toGrow:
            grow = toGrow.popleft()
            for u in g.neighbours(grow):
                if colour[u] == None:  # Not coloured yet
                    colour[u] = not colour[grow]  # Colour it the "other" colour from this one
                    toGrow.append(u)
                else:
                    if colour[u] == colour[grow]:
                        # Whoops, we cannot colour this.
                        return False
    # Coloured all the connected vertices in a satisfactory manner
    return True


def topSort1(g, v):
    """Sorts the vertices in DirectedGraph g in a topological order

    This function performs a depth-first search of g, starting at v, to determine
    the dependencies and then returns a topological ordering of the vertices
    such that for every vertex v, its ancestors in g appear earlier in the
    ordering.

    The ordering is represented by a list with element i being the order of
    vertex i.  If vertex i is not discoverable by a depth-first search from v,
    element i of this list will be None.

    g must be a directed acyclic graph, or an error is raised.

    If g is not a connected graph, it may be better to use topSort2 instead.

    @param g: the graph to sort
    @type g: graphadt.graphtypes.Graph
    @param v: the label of a vertex
    @type v: integer

    @rtype: is of zero or more integers and zero or more None
    @returns: zero or more integers and zero or more None

    @raises ValueError: if g is not a directed acyclic graph
    @raises ValueError: if v is not a vertex in g
    """

    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")

    if not isinstance(g, DirectedGraph):
        raise ValueError("G must be a directed graph")

    if not isAcyclic(g):
        raise ValueError("G must be an acyclic graph")

    sort = [None] * g.order()

    level, preorder, postorder = DFS(g, v)

    for i in range(len(postorder)):
        if postorder[i] is not None:
            sort[i] = level - postorder[i]
    return sort


def topSort2(g):
    """Sorts the vertices in DirectedGraph g in a topological order

    This function returns a topological ordering of the vertices such that for
    every vertex v, its ancestors in g appear earlier in the ordering.

    The ordering is represented by a list with element i being the order of
    vertex i.

    g must be a directed acyclic graph, or an error is raised.

    @param g: the graph to sort
    @type g: graphadt.graphtypes.Graph

    @rtype: is of zero or more integers and zero or more None
    @returns: zero or more integers and zero or more None

    @raises ValueError: if g is not a directed acyclic graph
    """
    if not isinstance(g, DirectedGraph):
        raise ValueError("G must be a directed graph")

    if not isAcyclic(g):
        raise ValueError("G must be an acyclic graph")

    n = g.order()
    sort = [None] * n
    inDeg = [g.inDegree(i) for i in range(n)]
    count = 0

    progress = True
    while progress:
        progress = False

        for v in range(n):
            # If this node is now precedent free,
            # And has not been sorted
            if inDeg[v] == 0 and sort[v] == None:
                sort[v] = count
                count += 1
                progress = True

                for i in g.neighbours(v):
                    inDeg[i] -= 1
    return sort


def maxDistance(g, v):
    """Returns the length to the shortest path to the furthest vertex from v, as
    well as the distance to all the other vertices.

    This function computes the eccentricity of vertex v in graph g by
    performing breadth-first search and noting the distance of the last
    node to be discovered.  This function also computes the distance of every
    vertex discovered in this way.  Vertices that cannot be discovered are set
    to a distance of infinity.  If all nodes are not discovered, the
    eccentricity of v is set to infinity.

    This function returns a tuple where the first element is the eccentricity of
    v.  The second element is a list, where element i is the shortest distance
    from vertex v to vertex i.

    @param g: the graph
    @type g: graphadt.graphtypes.Graph
    @param v: the label of a vertex
    @type v: integer

    @returns: eccentricity of v, distance of other vertices from v
    @rtype: tuple -> integer or infinity, list

    @raises ValueError: if v is not the label of a vertex in g
    """
    n = g.order()
    if not g.order() > v >= 0:
        raise ValueError("Value of v out of range")

    dist = [float("infinity")] * n  # Create an n-long vector filled with n
    # I.e., set all to maximum distance
    depth = 0
    dist[v] = depth
    distlist = [v]
    count = 1  # Number of discovered vertices
    while distlist:
        depth += 1
        nextlist = []
        for i in distlist:
            for j in g.neighbours(i):
                if isinf(dist[j]):  # First time we're seeing this vertex
                    dist[j] = depth
                    count += 1  # We've discovered one more vertex
                    nextlist.append(j)
        distlist = nextlist
    if count == n:  # If we've found them all
        return depth - 1, dist  # Return the depth of the last vertex
    else:
        return float("infinity"), dist  # Else return infinity


def distanceMatrix(g):
    """Returns a matrix of all the distances between vertices in g.

    This function returns an n x n matrix, where n is the size of the vertex set
    of g.  The value at element i,j of this matrix is the shortest distance from
    vertex i to vertex j.  If j is not reachable from i, then the element at i,j
    is set to infinity.

    @param g: the graph
    @type g: graphadt.graphtypes.Graph

    @returns: distance matrix of g
    @rtype: list of lists
    """

    matrix = []
    for v in range(g.order()):
        matrix.append(maxDistance(g, v)[1])
    return matrix


def diameter(g):
    """Returns the diameter of g.

    The diameter of g is defined as the maximum eccentricity of its vertices.
    This function calculates the eccentricity of every vertex in g and then
    returns the maximum.

    @param g: the graph
    @type g: graphadt.graphtypes.Graph

    @returns: the diameter of g
    @rtype: integer or infinity

    """

    largestDistance = 0
    for v in range(g.order()):
        d, dist = maxDistance(g, v)
        largestDistance = max(largestDistance, d)
    return largestDistance


def radius(g):
    """Returns the radius of g.

    The radius of g is defined as the minimum eccentricity of its vertices.
    This function calculates the eccentricity of every vertex in g and then
    returns the minimum.

    @param g: the graph
    @type g: graphadt.graphtypes.Graph

    @returns: the radius of g
    @rtype: integer or infinity
    """

    smallestDistance = float("inf")
    if g.order() == 0:  # A graph with no vertices has 0
        return 0  # radius
    for v in range(g.order()):
        d, dist = maxDistance(g, v)
        smallestDistance = min(smallestDistance, d)
    return smallestDistance
