package linearProgramming;

import lpsolve.LpSolveException;

/**
 * Contains minimum and maximum values.
 */
public class Variable {

	public LinearSpec ls;
	double min = 0;
	double max = Double.MAX_VALUE;

	/**
	 * Gets index of the variable.
	 * 
	 * @return the index of the variable
	 */
	public int getIndex() throws Exception {
		int i = ls.variables.indexOf(this);
		if (i == -1) throw new Exception("Variable not part of ls.variables.");
		return i + 1;
	}

	public double value = Double.NaN;

	/**
	 * Gets the minimum value of the variable.
	 * 
	 * @return the minimum value of variable
	 */
	public double getMin() {
		return min;
	}

	/**
	 * Sets the minimum value of the variable.
	 * 
	 * @param value	minimum value
	 */
	public void setMin(double value) throws Exception {
		min = value;
		ls.lp.setBounds(this.getIndex(), min, max);
	}

	/**
	 * Gets the maximum value of the variable.
	 * 
	 * @return the maximum value of variable
	 */
	public double getMax() {
		return max;
	}

	/**
	 * Sets the maximum value of the variable.
	 * 
	 * @param value	maximum value
	 */
	public void setMax(double value) throws Exception {
		max = value;
		ls.lp.setBounds(this.getIndex(), min, max);
	}

	/**
	 * Sets the minimum and maximum values of the variable.
	 * 
	 * @param min	minimum value
	 * @param max	maximum value
	 */
	public void setRange(double min, double max) throws Exception {
		this.min = min;
		this.max = max;
		ls.lp.setBounds(this.getIndex(), min, max);
	}

	/**
	 * Returns index of the variable as <code>String</code>.
	 * E.g. "Var2"
	 * 
	 * @return the <code>String</code> index of the variable
	 */
	public String toString() {
		try {
			return "Var" + getIndex();
		} catch (Exception e) {
			return "Exception in getIndex().";
		}
	}

	/**
	 * Adds a constraint that sets this variable equal to the given one.
	 * 
	 * @param v	variable that should have the same value
	 * @return the new equality constraint
	 */
	public Constraint isEqual(Variable v) throws Exception {
		return ls.addConstraint(1.0, this, -1.0, v, OperatorType.EQ, 0);
	}

	/**
	 * Adds a constraint that sets this variable smaller or equal to the given one.
	 * 
	 * @param v	variable that should have a larger or equal value
	 * @return the new constraint
	 */
	public Constraint isSmallerOrEqual(Variable v) throws Exception {
		return ls.addConstraint(1.0, this, -1.0, v, OperatorType.LE, 0);
	}

	/**
	 * Adds a constraint that sets this variable greater or equal to the given one.
	 * 
	 * @param v	variable that should have a smaller or equal value
	 * @return the new constraint
	 */
	public Constraint isGreaterOrEqual(Variable v) throws Exception {
		return ls.addConstraint(-1.0, v, 1.0, this, OperatorType.GE, 0);
	}

	protected Variable(LinearSpec ls) {
		this.ls = ls;
		ls.variables.add(this);
		if (ls.variables.size() > ls.columns) {
			double[] d = {0};
			int[] i = {0};
			try {
				ls.lp.addColumnex(0, d, i);
			} catch (LpSolveException e) {
				System.err.println("Error in addColumnex.");
			}
		}
	}

	/**
	 * Removes the variable from its specification.
	 */
	public void remove() throws Exception {
		ls.lp.delColumn(this.getIndex());
		ls.variables.remove(this);
	}
}
