package linearProgramming;

import java.util.*;
import lpsolve.*;

/**
 * Specification of a linear programming problem.
 */
public class LinearSpec {

	public LpSolve lp;
	LpSolve lpPresolved = null;
	int columns = 0;
	OptimizationType optimization = OptimizationType.MINIMIZE;

	/**
	 * The summands of the objective function.
	 */
	public List<Summand> objFunctionSummands = new ArrayList<Summand>();

	/**
	 * The variables of the specification.
	 */
	public List<Variable> variables = new ArrayList<Variable>();

	/**
	 * The constraints of the specification.
	 */
	public List<Constraint> constraints = new ArrayList<Constraint>();

	/**
	 * The result of the last solving attempt.
	 */
	public ResultType result = ResultType.ERROR;

	/**
	 * The value of the objective function at the end of the last solving attempt.
	 */
	public double objectiveValue = Double.NaN;

	/**
	 * The duration of the last solving attempt in milliseconds.
	 */
	public double solvingTime = Double.NaN;

	/**
	 * Creates a new specification for a linear programming problem.
	 */
	public LinearSpec() throws Exception {
		lp = LpSolve.makeLp(0, 0);
		if (lp.getLp() == 0) throw new Exception("Couldn't construct a new model.");
		lp.setVerbose(1); // only print critical messages
	}

	public void updateObjFunction() throws Exception {
		double[] coeffs = new double[objFunctionSummands.size()];
		int[] vindexes = new int[objFunctionSummands.size()];
		int i = 0;
		for (Summand s : objFunctionSummands) {
			coeffs[i] = s.getCoeff();
			vindexes[i] = s.getVar().getIndex();
			i++;
		}
		try {
			lp.setObjFnex(objFunctionSummands.size(), coeffs, vindexes);
		} catch (LpSolveException e) {
			System.err.println("Error in setObjFnex.");
		}
		removePresolved();
	}

	/**
	 * Sets the objective function.
	 * 
	 * @param coeffs	the objective function's coefficients
	 * @param vars		the objective function's variables
	 */
	public void setObjFunction(double[] coeffs, Variable[] vars) throws Exception {
		if (coeffs.length != vars.length)
			throw new Exception("Number of coefficients and number of variables in objective function must be equal.");

		objFunctionSummands = new ArrayList<Summand>();
		for(int i=0; i<coeffs.length; i++)
			objFunctionSummands.add(new Summand(coeffs[i], vars[i]));

		updateObjFunction();
	}

	/**
	 * Adds a new summand to the objective function.
	 * 
	 * @param coeff	the summand's coefficient
	 * @param var	the summand's variable
	 * @return the new summand
	 */
	public Summand addObjFunctionSummand(double coeff, Variable var) throws Exception {
		Summand s = new Summand(coeff, var);
		objFunctionSummands.add(s);
		updateObjFunction();
		return s;
	}

	/**
	 * Gets the current optimization.
	 * The default is minimization.
	 * 
	 * @return the current optimization
	 */
	public OptimizationType getOptimization() {
		return optimization;
	}

	/**
	 * Sets whether the solver should minimize or maximize the objective function.
	 * The default is minimization.
	 * 
	 * @param optimization	the optimization type
	 */
	public void setOptimization(OptimizationType optimization) { 
		if (optimization == OptimizationType.MINIMIZE)
			lp.setMinim();
		else
			lp.setMaxim();

		removePresolved();
	}

	/**
	 * Adds a new variable to the specification.
	 * 
	 * @return the new variable
	 */
	public Variable addVariable() throws LpSolveException {
		return new Variable(this);
	}

	/**
	 * Adds a new hard linear constraint to the specification.
	 * 
	 * @param coeffs		the constraint's coefficients
	 * @param vars			the constraint's variables
	 * @param op			the constraint's operand
	 * @param rightSide	the constant value on the constraint's right side
	 * @return the new constraint
	 */
	public Constraint addConstraint(Summand[] summands, OperatorType op, double rightSide) throws Exception {
		Constraint c = new Constraint(this, summands, op, rightSide,
				Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
		removePresolved();
		return c;
	}

	/**
	 * Adds a new hard linear constraint to the specification with a single summand.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param op			the constraint's operand
	 * @param rightSide	the constant value on the constraint's right side
	 * @return the new constraint
	 */
	public Constraint addConstraint(double coeff1, Variable var1, OperatorType op, double rightSide) throws Exception {
		Summand[] summands = new Summand[1];
		summands[0] = new Summand(coeff1, var1);
		Constraint c = new Constraint(this, summands, op, rightSide,
				Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new hard linear constraint to the specification with two summands.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param coeff2		the constraint's second coefficient
	 * @param var2			the constraint's second variable
	 * @param op			the constraint's operand
	 * @param rightSide	the constant value on the constraint's right side
	 * @return the new constraint
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			double coeff2, Variable var2, OperatorType op, double rightSide) throws Exception {
		Summand[] summands = new Summand[2];
		summands[0] = new Summand(coeff1, var1);
		summands[1] = new Summand(coeff2, var2);
		Constraint c = new Constraint(this, summands, op, rightSide,
				Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new hard linear constraint to the specification with three summands.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param coeff2		the constraint's second coefficient
	 * @param var2			the constraint's second variable
	 * @param coeff3		the constraint's third coefficient
	 * @param var3			the constraint's third variable
	 * @param op			the constraint's operand
	 * @param rightSide	the constant value on the constraint's right side
	 * @return the new constraint
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			double coeff2, Variable var2, double coeff3, Variable var3, 
			OperatorType op, double rightSide) throws Exception {
		Summand[] summands = new Summand[3];
		summands[0] = new Summand(coeff1, var1);
		summands[1] = new Summand(coeff2, var2);
		summands[2] = new Summand(coeff3, var3);
		Constraint c = new Constraint(this, summands, op, rightSide,
				Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new hard linear constraint to the specification with four summands.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param coeff2		the constraint's second coefficient
	 * @param var2			the constraint's second variable
	 * @param coeff3		the constraint's third coefficient
	 * @param var3			the constraint's third variable
	 * @param coeff4		the constraint's fourth coefficient
	 * @param var4			the constraint's fourth variable
	 * @param op			the constraint's operand
	 * @param rightSide	the constant value on the constraint's right side
	 * @return the new constraint
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			double coeff2, Variable var2, double coeff3, Variable var3,
			double coeff4, Variable var4, OperatorType op, double rightSide) throws Exception {
		Summand[] summands = new Summand[4];
		summands[0] = new Summand(coeff1, var1);
		summands[1] = new Summand(coeff2, var2);
		summands[2] = new Summand(coeff3, var3);
		summands[3] = new Summand(coeff4, var4);
		Constraint c = new Constraint(this, summands, op, rightSide,
				Double.POSITIVE_INFINITY, Double.POSITIVE_INFINITY);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new soft linear constraint to the specification.
	 * i.e. a constraint that does not always have to be satisfied.
	 * 
	 * @param coeffs		the constraint's coefficients
	 * @param vars			the constraint's variables
	 * @param op			the constraint's operand
	 * @param rightSide		the constant value on the constraint's right side
	 * @param penaltyNeg	the coefficient penalizing negative deviations from the exact solution
	 * @param penaltyPos	the coefficient penalizing positive deviations from the exact solution
	 */
	public Constraint addConstraint(Summand[] summands, OperatorType op, 
			double penaltyNeg, double penaltyPos, double rightSide) throws Exception {
		Constraint c = new Constraint(this, summands, op, rightSide,
				penaltyNeg, penaltyPos);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new soft linear constraint to the specification with a single summand.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param op			the constraint's operand
	 * @param rightSide		the constant value on the constraint's right side
	 * @param penaltyNeg	the coefficient penalizing negative deviations from the exact solution
	 * @param penaltyPos	the coefficient penalizing positive deviations from the exact solution
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			OperatorType op, double rightSide, double penaltyNeg, double penaltyPos) throws Exception {
		Summand[] summands = new Summand[1];
		summands[0] = new Summand(coeff1, var1);
		Constraint c = new Constraint(this, summands, op, rightSide,
				penaltyNeg, penaltyPos);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new soft linear constraint to the specification with two summands.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param coeff2		the constraint's second coefficient
	 * @param var2			the constraint's second variable
	 * @param op			the constraint's operand
	 * @param rightSide		the constant value on the constraint's right side
	 * @param penaltyNeg	the coefficient penalizing negative deviations from the exact solution
	 * @param penaltyPos	the coefficient penalizing positive deviations from the exact solution
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			double coeff2, Variable var2, OperatorType op, double rightSide, double penaltyNeg, double penaltyPos) throws Exception {
		Summand[] summands = new Summand[2];
		summands[0] = new Summand(coeff1, var1);
		summands[1] = new Summand(coeff2, var2);
		Constraint c = new Constraint(this, summands, op, rightSide,
				penaltyNeg, penaltyPos);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new soft linear constraint to the specification with three summands.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param coeff2		the constraint's second coefficient
	 * @param var2			the constraint's second variable
	 * @param coeff3		the constraint's third coefficient
	 * @param var3			the constraint's third variable
	 * @param op			the constraint's operand
	 * @param rightSide		the constant value on the constraint's right side
	 * @param penaltyNeg	the coefficient penalizing negative deviations from the exact solution
	 * @param penaltyPos	the coefficient penalizing positive deviations from the exact solution
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			double coeff2, Variable var2, double coeff3, Variable var3,
			OperatorType op, double rightSide, double penaltyNeg, double penaltyPos) throws Exception {
		Summand[] summands = new Summand[3];
		summands[0] = new Summand(coeff1, var1);
		summands[1] = new Summand(coeff2, var2);
		summands[2] = new Summand(coeff3, var3);
		Constraint c = new Constraint(this, summands, op, rightSide,
				penaltyNeg, penaltyPos);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new soft linear constraint to the specification with four summands.
	 * 
	 * @param coeff1		the constraint's first coefficient
	 * @param var1			the constraint's first variable
	 * @param coeff2		the constraint's second coefficient
	 * @param var2			the constraint's second variable
	 * @param coeff3		the constraint's third coefficient
	 * @param var3			the constraint's third variable
	 * @param coeff4		the constraint's fourth coefficient
	 * @param var4			the constraint's fourth variable
	 * @param op			the constraint's operand
	 * @param rightSide		the constant value on the constraint's right side
	 * @param penaltyNeg	the coefficient penalizing negative deviations from the exact solution
	 * @param penaltyPos	the coefficient penalizing positive deviations from the exact solution
	 */
	public Constraint addConstraint(double coeff1, Variable var1, 
			double coeff2, Variable var2, double coeff3, Variable var3,
			double coeff4, Variable var4, OperatorType op, double rightSide, double penaltyNeg, double penaltyPos) throws Exception {
		Summand[] summands = new Summand[4];
		summands[0] = new Summand(coeff1, var1);
		summands[1] = new Summand(coeff2, var2);
		summands[2] = new Summand(coeff3, var3);
		summands[3] = new Summand(coeff4, var4);
		Constraint c = new Constraint(this, summands, op, rightSide,
				penaltyNeg, penaltyPos);
		removePresolved();
		return c;
	}
	
	/**
	 * Adds a new penalty function to the specification.
	 * 
	 * @param var	the penalty function's variable
	 * @param xs	the penalty function's sampling points
	 * @param gs	the penalty function's gradients
	 * @return the new penalty function
	 */
	public PenaltyFunction addPenaltyFunction(Variable var, double[] xs, double[] gs) throws Exception {
		return new PenaltyFunction(this, var, xs, gs);
	}

	/**
	 * Remove a cached presolved model, if existent.
	 * This is automatically done each time after the model has been changed, 
	 * to avoid an old cached presolved model getting out of sync.
	 */
	public void removePresolved() {
		if (lpPresolved == null) return;
		lpPresolved.deleteLp();
		lpPresolved = null;
	}

	/**
	 * Creates and caches a simplified version of the linear programming problem, 
	 * where redundant rows, columns and constraints are removed, 
	 * if it has not been created before.
	 * Then, the simplified problem is solved.
	 * 
	 * @return the result of the solving attempt
	 */
	public ResultType presolve() throws Exception {
		long start, end;
		start = System.nanoTime();

		if (lpPresolved == null) {
			lpPresolved = lp.copyLp();
			lpPresolved.setPresolve(
					LpSolve.PRESOLVE_ROWS
					| LpSolve.PRESOLVE_COLS
					| LpSolve.PRESOLVE_LINDEP,
					lpPresolved.getPresolveloops());
		}

		result = ResultType.getResultType(lpPresolved.solve());
		objectiveValue = lpPresolved.getObjective();

		if (result == ResultType.OPTIMAL) {
			for (Variable v : variables)
				v.value = lpPresolved.getVarPrimalresult(lpPresolved.getNorigRows() + v.getIndex());
		}

		end = System.nanoTime();
		solvingTime = (double)(end - start) * 1000.0;
		return result;
	}

	/**
	 * Tries to solve the linear programming problem.
	 * If a cached simplified version of the problem exists, it is used instead.
	 * 
	 * @return the result of the solving attempt
	 */
	public ResultType solve() throws Exception {
		if (lpPresolved != null) return presolve();

		long start, end;
		start = System.nanoTime();

		result = ResultType.getResultType(lp.solve());
		objectiveValue = lp.getObjective();

		if (result == ResultType.OPTIMAL) {
			double[] x = new double[variables.size()];
			try {
				lp.getVariables(x);
			} catch (LpSolveException e) {
				System.err.println("Error in getVariables.");
			}
			int i = 0;
			for (Variable v : variables) v.value = x[i++];
		}
		end = System.nanoTime();
		solvingTime = (double)(end - start) * 1000.0;
		return result;
	}

	/**
	 * Writes the specification into a text file.
	 * The file will be overwritten if it exists.
	 * 
	 * @param fname	the file name
	 */
	public void save(String fname) throws LpSolveException {
		lp.writeLp(fname);
	}

	/**
	 * Removes the specification.
	 */
	public void remove() {
		lp.deleteLp();
	}
}
