package linearProgramming;

import lpsolve.*;

/**
 * Hard linear constraint, i.e.&nbsp;one that must be satisfied. 
 * May render a specification infeasible.
 */
public class Constraint {
	LinearSpec ls;
	Summand[] leftSide;
	OperatorType op;
	double rightSide;
	Summand dNegSummand, dPosSummand;

	/**
	 * Gets the index of the constraint.
	 * 
	 * @return the index of the constraint
	 */
	public int getIndex() throws Exception {
		int i = ls.constraints.indexOf(this);
		if (i == -1) throw new Exception("Constraint not part of ls.constraints.");
		return i + 1;
	}
	
	/**
	 * Gets the left side of the constraint.
	 * 
	 * @return the summands on the left side of the constraint
	 */
	public Summand[] getLeftSide() {
		return leftSide;
	}
	
	/**
	 * Sets the summands on the left side of the constraint.
	 * The old summands are NOT deleted.
	 * 
	 * @param summands	an array of Summand objects that make up the new left side
	 */
	public void setLeftSide(Summand[] summands) throws Exception {
		leftSide = summands;
		updateLeftSide();
	}

	public void updateLeftSide() throws Exception {
		double[] coeffs = new double[leftSide.length + 2];
		int[] vindexes = new int[leftSide.length + 2];
		
		int i = 0;
		
		for (Summand s : leftSide) {
			coeffs[i] = s.getCoeff();
			vindexes[i] = s.getVar().getIndex();
			i++;
		}

		if ((dNegSummand != null) && (op != OperatorType.LE)) {
			vindexes[i] = dNegSummand.getVar().getIndex();
			coeffs[i] = 1.0;
			i++;
		}
		
		if ((dPosSummand != null) && (op != OperatorType.GE)) {
			vindexes[i] = dPosSummand.getVar().getIndex();
			coeffs[i] = -1.0;
			i++;
		}

		try {
			ls.lp.setRowex(this.getIndex(), i, coeffs, vindexes);
		} catch (LpSolveException e) {
			System.err.println("Error in setRowex.");
		}
		
		ls.updateObjFunction();
		ls.removePresolved();
	}
	
	public void setLeftSide(double coeff1, Variable var1) throws Exception {
		for (int i = 0; i < leftSide.length; i++)
			leftSide[i].remove();
		leftSide = new Summand[1];
		leftSide[0] = new Summand(coeff1, var1);
		updateLeftSide();
	}
	
	public void setLeftSide(double coeff1, Variable var1, double coeff2, Variable var2) throws Exception {
		for (int i = 0; i < leftSide.length; i++)
			leftSide[i].remove();
		leftSide = new Summand[2];
		leftSide[0] = new Summand(coeff1, var1);
		leftSide[1] = new Summand(coeff2, var2);
		updateLeftSide();
	}
	
	public void setLeftSide(double coeff1, Variable var1, double coeff2, Variable var2, 
			double coeff3, Variable var3) throws Exception {
		for (int i = 0; i < leftSide.length; i++)
			leftSide[i].remove();
		leftSide = new Summand[3];
		leftSide[0] = new Summand(coeff1, var1);
		leftSide[1] = new Summand(coeff2, var2);
		leftSide[2] = new Summand(coeff3, var3);
		updateLeftSide();
	}
	
	public void setLeftSide(double coeff1, Variable var1, double coeff2, Variable var2, 
			double coeff3, Variable var3, double coeff4, Variable var4) throws Exception {
		for (int i = 0; i < leftSide.length; i++)
			leftSide[i].remove();
		leftSide = new Summand[4];
		leftSide[0] = new Summand(coeff1, var1);
		leftSide[1] = new Summand(coeff2, var2);
		leftSide[2] = new Summand(coeff3, var3);
		leftSide[3] = new Summand(coeff4, var4);
		updateLeftSide();
	}
	
	/**
	 * Gets the operator used for this constraint.
	 * 
	 * @return the operator used for this constraint
	 */
	public OperatorType getOp() {
		return op;
	}
	
	/**
	 * Sets the operator used for this constraint.
	 * 
	 * @param value	operator
	 */
	public void setOp(OperatorType value) throws Exception {
		op = value;
		try {
			ls.lp.setConstrType(this.getIndex(),
					((op == OperatorType.EQ) ? LpSolve.EQ
					: (op == OperatorType.GE) ? LpSolve.GE
					: LpSolve.LE));
		} catch (LpSolveException e) {
			System.err.println("Error in setConstrType.");
		}
		ls.removePresolved();
	}

	/**
	 * Gets the constant value that is on the right side of the operator.
	 * 
	 * @return the constant value that is on the right side of the operator
	 */
	public double getRightSide() {
		return rightSide;
	}

	/**
	 * Sets the constant value that is on the right side of the operator.
	 * 
	 * @param value	constant value that is on the right side of the operator
	 */
	public void setRightSide(double value) throws Exception {
		rightSide = value;
		try {
			ls.lp.setRh(this.getIndex(), rightSide);
		} catch (LpSolveException e) {
			System.err.println("Error in setRh.");
		}
		ls.removePresolved();
	}
	
	/**
	 * Gets the coefficient of negative summand.
	 * 
	 * @return the coefficient of negative summand.
	 */
	public double getPenaltyNeg() {
		if (dNegSummand == null)
			return Double.POSITIVE_INFINITY;
		return dNegSummand.getCoeff();
	}

	/**
	 * The penalty coefficient for positive deviations from the soft constraint's exact solution,&nbsp;
	 * i.e. if the left side is too large.
	 * 
	 * @param value	coefficient of negative penalty <code>double</code>
	 */
	public void setPenaltyNeg(double value) throws Exception {
		if (dNegSummand == null) {
			dNegSummand = new Summand(value, new Variable(ls));
			ls.objFunctionSummands.add(dNegSummand);
			updateLeftSide();
			ls.updateObjFunction();
			return;
		}
		
		if (value == dNegSummand.getCoeff()) return;
		dNegSummand.setCoeff(value);
		ls.updateObjFunction();
	}
	
	/**
	 * Gets the coefficient of positive summand.
	 * 
	 * @return the coefficient of positive summand.
	 */
	public double getPenaltyPos() {
		if (dPosSummand == null)
			return Double.POSITIVE_INFINITY;
		return dPosSummand.getCoeff();
	}

	/**
	 * The penalty coefficient for negative deviations from the soft constraint's exact solution,
	 * i.e. if the left side is too small.
	 * 
	 * @param value	coefficient of positive penalty <code>double</code>
	 */
	public void setPenaltyPos(double value) throws Exception {
		if (dPosSummand == null) {
			dPosSummand = new Summand(value, new Variable(ls));
			ls.objFunctionSummands.add(dPosSummand);
			updateLeftSide();
			ls.updateObjFunction();
			return;
		}
		
		if (value == dPosSummand.getCoeff()) return;
		dPosSummand.setCoeff(value);
		ls.updateObjFunction();
	}
	
	/**
	 * Gets the slack variable for the negative variations.
	 * 
	 * @return the slack variable for the negative variations
	 */
	public Variable getDNeg() {
		if (dNegSummand == null)
			return null;
		return dNegSummand.getVar();
	}
	
	/**
	 * Gets the slack variable for the positive variations.
	 * 
	 * @return the slack variable for the positive variations
	 */
	public Variable getDPos() {
		if (dPosSummand == null)
			return null;
		return dPosSummand.getVar();
	}
	
	/**
	 * Constructor.
	 */
	Constraint(LinearSpec ls, Summand[] summands, OperatorType op, double rightSide,
			double penaltyNeg, double penaltyPos) throws Exception {

		this.ls = ls;
		this.leftSide = summands;
		this.op = op;
		this.rightSide = rightSide;

		double[] coeffs = new double[summands.length + 2];
		int[] vindexes = new int[summands.length + 2];
		int i = 0;
		for (Summand s : summands) {
			coeffs[i] = s.getCoeff();
			vindexes[i] = s.getVar().getIndex();
			i++;
		}
		
		if ((penaltyNeg != Double.POSITIVE_INFINITY)
				&& (op != OperatorType.LE)) {
			dNegSummand = new Summand(penaltyNeg, new Variable(ls));
			ls.objFunctionSummands.add(dNegSummand);
			vindexes[i] = dNegSummand.getVar().getIndex();
			coeffs[i] = 1.0;
			i++;
		}
		
		else
			dNegSummand = null;

		try {
			ls.lp.addConstraintex(i, coeffs, vindexes,
					((op == OperatorType.EQ) ? LpSolve.EQ
					: (op == OperatorType.GE) ? LpSolve.GE
					: LpSolve.LE),
					rightSide);
		} catch (LpSolveException e) {
			System.err.println("Error in addConstraintex.");
		}

		this.ls.updateObjFunction();
		ls.constraints.add(this);
	}

	/**
	 * Removes the constraint from its specification.
	 */
	public void remove() throws Exception {
		for (Summand s : leftSide)
			s.remove();
		if (dNegSummand != null) {
			dNegSummand.getVar().remove();
			dNegSummand.remove();
		}
		if (dPosSummand != null) {
			dPosSummand.getVar().remove();
			dPosSummand.remove();
		}
		ls.lp.delConstraint(this.getIndex());
		ls.constraints.remove(this);
	}
}
