package alm;

import java.util.*;
import javax.swing.*;
import java.awt.Dimension;
import java.io.FileWriter;

import linearProgramming.LinearSpec;
import linearProgramming.Summand;
import linearProgramming.ResultType;
import lpsolve.LpSolveException;

/**
 * Layout specification.
 */
public class LayoutSpec extends LinearSpec {
	/**
	 * The areas that were added to the specification.
	 */
	public List<Area> areas = new ArrayList<Area>();

	/**
	 * X-tab for the left border of the GUI.
	 */
	public XTab left;

	/**
	 * X-tab for the right border of the GUI.
	 */
	public XTab right;

	/**
	 * Y-tab for the top border of the GUI.
	 */
	public YTab top;

	/**
	 * Y-tab for the bottom border of the GUI.
	 */
	public YTab bottom;

	/**
	 * Constructor for class <code>LayoutSpec</code>.
	 */
	public LayoutSpec() throws Exception {
		super();
		left = new XTab(this);
		right = new XTab(this);
		top = new YTab(this);
		bottom = new YTab(this);
		
		// the Left tab is always at x-position 0, and the Top tab is always at y-position 0
		left.setRange(0, 0);
		top.setRange(0, 0);
	}
	
	public void solveLayout() throws LpSolveException, Exception {
		// if autoPrefContentSize is set on an area, readjust its prefContentSize and rigidity settings
		for (Area a : areas)
			if (a.autoPrefContentSize) a.setDefaultPrefContentSize();

		// try to solve the layout until the result is OPTIMAL or INFEASIBLE, maximally 15 tries
		// sometimes the solving algorithm encounters numerical problems (NUMFAILURE), and
		// repeating the solving often helps to overcome them
		FileWriter sw = new FileWriter("performance.txt", true);
		ResultType result = ResultType.ERROR;
		for (int tries = 0; tries < 15; tries++) {
			result = solve();
			sw.write(result + "\t" + solvingTime + "ms"
					+ "\t#vars=" + variables.size() + "\t#constraints=" + constraints.size() + "\n");
			if (result == ResultType.OPTIMAL || result == ResultType.INFEASIBLE) break;
		}
		sw.close();
	}
	
	// cached layout values
	// need to be invalidated whenever the layout specification is changed
	Dimension minSize = Area.UNDEFINED_SIZE;
	Dimension maxSize = Area.UNDEFINED_SIZE;
	Dimension preferredSize = Area.UNDEFINED_SIZE;
	
	public void invalidateLayout() {
		minSize = Area.UNDEFINED_SIZE;
		maxSize = Area.UNDEFINED_SIZE;
		preferredSize = Area.UNDEFINED_SIZE;
	}
	
	public Dimension getMinSize() throws Exception {
		if (minSize == Area.UNDEFINED_SIZE)
			minSize = calculateMinSize();
		return minSize;
	}
	
	public Dimension getMaxSize() throws Exception {
		if (maxSize == Area.UNDEFINED_SIZE)
			maxSize = calculateMaxSize();
		return maxSize;
	}
	
	public Dimension getPreferredSize() throws Exception {
		if (preferredSize == Area.UNDEFINED_SIZE)
			preferredSize = calculatePreferredSize();
		return preferredSize;
	}
	
	Dimension calculateMinSize() throws Exception {
		List<Summand> buf = objFunctionSummands;
		objFunctionSummands = new ArrayList<Summand>();
		addObjFunctionSummand(1, right);
		addObjFunctionSummand(1, bottom);
		solveLayout();
		objFunctionSummands = buf;
		updateObjFunction();
		
		if (result == ResultType.UNBOUNDED)
			return Area.MIN_SIZE;
		if (result != ResultType.OPTIMAL) {
			save("failed-layout.txt");
            throw new Exception("Could not solve the layout specification (" 
            		+ result.toString() + "). Saved specification in file failed-layout.txt");
		}
		
		return new Dimension((int)(right.value - left.value), (int)(bottom.value - top.value));
	}
	
	Dimension calculateMaxSize() throws Exception {
		List<Summand> buf = objFunctionSummands;
		objFunctionSummands = new ArrayList<Summand>();
		addObjFunctionSummand(-1, right);
		addObjFunctionSummand(-1, bottom);
		solveLayout();
		objFunctionSummands = buf;
		updateObjFunction();
		
		if (result == ResultType.UNBOUNDED)
			return Area.MAX_SIZE;
		if (result != ResultType.OPTIMAL) {
			save("failed-layout.txt");
            throw new Exception("Could not solve the layout specification (" 
            		+ result.toString() + "). Saved specification in file failed-layout.txt");
		}
		
		return new Dimension((int)(right.value - left.value), (int)(bottom.value - top.value));
	}
	
	Dimension calculatePreferredSize() throws Exception {
		solveLayout();
		if (result != ResultType.OPTIMAL) {
			save("failed-layout.txt");
            throw new Exception("Could not solve the layout specification (" 
            		+ result.toString() + "). Saved specification in file failed-layout.txt");
		}
		
		return new Dimension((int)(right.value - left.value), (int)(bottom.value - top.value));
	}

	/**
	 * Adds a new x-tab to the specification.
	 * 
	 * @return the new x-tab
	 */
	public XTab addXTab() {
		return new XTab(this);
	}

	/**
	 * Adds a new y-tab to the specification.
	 * 
	 * @return the new y-tab
	 */
	public YTab addYTab() {
		return new YTab(this);
	}

	/**
	 * Adds a new row to the specification.
	 * 
	 * @return the new row
	 */
	public Row addRow() {
		return new Row(this);
	}

	/**
	 * Adds a new row to the specification that is glued to the given y-tabs.
	 * 
	 * @param top
	 * @param bottom
	 * @return the new row
	 */
	public Row addRow(YTab top, YTab bottom) throws Exception {
		Row row = new Row(this);
		if (top != null) row.constraints.add(row.getTop().isEqual(top));
		if (bottom != null) row.constraints.add(row.getBottom().isEqual(bottom));
		return row;
	}
	
	/**
	 * Adds a new column to the specification.
	 * 
	 * @return the new column
	 */
	public Column addColumn() {
		return new Column(this);
	}
	
	/**
	 * Adds a new column to the specification that is glued to the given x-tabs.
	 * 
	 * @param left
	 * @param right
	 * @return the new column
	 */
	public Column addColumn(XTab left, XTab right) throws Exception {
		Column column = new Column(this);
		if (left != null) column.constraints.add(column.getLeft().isEqual(left));
		if (right != null) column.constraints.add(column.getRight().isEqual(right));
		return column;
	}
	
	/**
	 * Adds a new area to the specification, setting only the necessary minimum size constraints.
	 * 
	 * @param left				left border
	 * @param top				top border
	 * @param right				right border
	 * @param bottom			bottom border
	 * @param content			the control which is the area content
	 * @param minContentSize	minimum content size
	 * @return the new area
	 */
	public Area addArea(XTab left, YTab top, XTab right, YTab bottom, JComponent content, 
			Dimension minContentSize) throws Exception {
		
		invalidateLayout();
		Area a = new Area(this, left, top, right, bottom, content, minContentSize);
		areas.add(a);
		return a;
	}

	/**
	 * Adds a new area to the specification, setting only the necessary minimum size constraints.
	 * 
	 * @param row				the row that defines the top and bottom border
	 * @param column			the column that defines the left and right border
	 * @param content			the control which is the area content
	 * @param minContentSize	minimum content size
	 * @return the new area
	 */
	public Area addArea(Row row, Column column, JComponent content, 
			Dimension minContentSize) throws Exception {
		
		invalidateLayout();
		Area a = new Area(this, row, column, content, minContentSize);
		areas.add(a);
		return a;
	}

	/**
	 * Adds a new area to the specification, automatically setting preferred size constraints.
	 * 
	 * @param left			left border
	 * @param top			top border
	 * @param right			right border
	 * @param bottom		bottom border
	 * @param content		the control which is the area content
	 * @return the new area
	 */
	public Area addArea(XTab left, YTab top, XTab right, YTab bottom, JComponent content) throws LpSolveException, Exception {
		invalidateLayout();
		Area a = new Area(this, left, top, right, bottom, content, new Dimension(0, 0));
		a.setDefaultPrefContentSize();
		a.autoPrefContentSize = true;
		areas.add(a);
		return a;
	}

	/**
	 * Adds a new area to the specification, automatically setting preferred size constraints.
	 * 
	 * @param row			the row that defines the top and bottom border
	 * @param column		the column that defines the left and right border
	 * @param content		the control which is the area content
	 * @return the new area    
	 */
	public Area addArea(Row row, Column column, JComponent content) throws Exception {
		invalidateLayout();
		Area a = new Area(this, row, column, content, new Dimension(0, 0));
		a.setDefaultPrefContentSize();
		a.autoPrefContentSize = true;
		areas.add(a);
		return a;
	}

	/**
	 * Finds the area that contains the given control.
	 * 
	 * @param c	the control to look for
	 * @return the area that contains the control
	 */
	public Area areaOf(JComponent c) {
		for (Area a : areas)
			if (a.getContent() == c) return a;
		return null;
	}
}
