package alm;

import java.util.*;
import javax.swing.*;
import java.awt.Point;
import java.awt.Dimension;
import linearProgramming.*;
import lpsolve.LpSolveException;

/**
 * Rectangular area in the GUI, defined by a tab on each side.
 */
public class Area {
	/**
	 * Minimum possible size. Use this if there is no lower bound.
	 */
	public static Dimension MIN_SIZE = new Dimension(0, 0);
	
	/**
	 * Maximum possible size. Use this if there is no upper bound.
	 */
	public static Dimension MAX_SIZE = new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);

	/**
	 * Undefined size. Used if a certain size constraint is not set.
	 */
	public static Dimension UNDEFINED_SIZE = new Dimension(-1, -1);

	/**
	 * The layout specification this area belongs to.
	 */
	LayoutSpec ls;

	XTab left, right;
	YTab top, bottom;
	Row row;
	Column column;
	JComponent content;

	/**
	 * A list of constraints which are removed form the speciifcation when the area is removed.
	 */
	List<Constraint> constraints = new ArrayList<Constraint>();

	Dimension minContentSize = new Dimension(0, 0);
	Dimension maxContentSize = MAX_SIZE;
	
	/**
	 * Size constraint for the content. Valid even if the content is actually in a child area.
	 */
	Constraint minContentWidth, maxContentWidth, minContentHeight, maxContentHeight;

	Dimension prefContentSize = UNDEFINED_SIZE;
	Dimension shrinkRigidity = new Dimension(2, 2);
	Dimension expandRigidity = new Dimension(1, 1);

	double contentAspectRatio = Double.NaN;
	Constraint contentAspectRatioC;

	/**
	 * Determines if the PrefContentSize is set automatically.
	 * Manual changes of PrefContentSize are ignored unless set to false.
	 */
	public boolean autoPrefContentSize = false;

	/**
	 * The soft constraints used for setting a preferred width.
	 */
	Constraint prefContentWidth;

	/**
	 * The soft constraints used for setting a preferred height.
	 */
	Constraint prefContentHeight;

	/**
	 * This area is used if the borders of the control are not the same as the borders of the area,
	 * i.e. if the control is smaller than the area due to margins or alignment.
	 * In case a childArea is present, many of the operations (such as those for setting content sizes)
	 * are delegated to the childArea.
	 */
	Area childArea;

	HorizontalAlignment hAlignment = HorizontalAlignment.FILL;
	VerticalAlignment vAlignment = VerticalAlignment.FILL;
	int leftMargin = 0;
	int topMargin = 0;
	int rightMargin = 0;
	int bottomMargin = 0;

	/**
	 * The constraint for setting margin and alignment on the left.
	 */
	Constraint leftConstraint;

	/**
	 * The constraint for setting margin and alignment at the top.
	 */
	Constraint topConstraint;

	/**
	 * The constraint for setting margin and alignment on the right.
	 */
	Constraint rightConstraint;

	/**
	 * The constraint for setting margin and alignment at the bottom.
	 */
	Constraint bottomConstraint;

	/**
	 * Left tab of the area.
	 */
	public XTab getLeft() { return left; }

	public void setLeft(XTab value) throws Exception {
		left = value;
		column = null; // since we changed an individual x-tab we do not align to a Column anymore
		
		if (childArea == null) {
			// the respective minimum constraint needs to use the new tab
			minContentWidth.setLeftSide(-1, left, 1, right);
			
			// if a maximum constraint was set, then it nees to use the new tab
			if (maxContentWidth != null)
				maxContentWidth.setLeftSide(-1, left, 1, right);
		} else
			updateHorizontal(); // the constraints between outer area and childArea need to use the new tab
		ls.invalidateLayout();
	}

	/**
	 * Right tab of the area.
	 */
	public XTab getRight() { return right; }

	public void setRight(XTab value) throws Exception {
		right = value;
		column = null; // since we changed an individual x-tab we do not align to a Column anymore
		
		if (childArea == null) {
			// the respective minimum constraint needs to use the new tab
			minContentWidth.setLeftSide(-1, left, 1, right);
			
			// if a maximum constraint was set, then it nees to use the new tab
			if (maxContentWidth != null)
				maxContentWidth.setLeftSide(-1, left, 1, right);
		} else
			updateHorizontal(); // the constraints between outer area and childArea need to use the new tab
		ls.invalidateLayout();
	}

	/**
	 * Top tab of the area.
	 */
	public YTab getTop() { return top; }

	public void setTop(YTab value) throws Exception {
		top = value;
		row = null; // since we changed an individual y-tab we do not align to a Row anymore
		
		if (childArea == null) {
			// the respective minimum constraint needs to use the new tab
			minContentHeight.setLeftSide(-1, top, 1, bottom);

			// if a maximum constraint was set, then it nees to use the new tab
			if (maxContentHeight != null)
				maxContentHeight.setLeftSide(-1, top, 1, bottom);
		} else
			updateVertical(); // the constraints between outer area and childArea need to use the new tab
		ls.invalidateLayout();
	}

	/**
	 * Bottom tab of the area.
	 */
	public YTab getBottom() { return bottom; }

	public void setBottom(YTab value) throws Exception {
		bottom = value;
		row = null; // since we changed an individual y-tab we do not align to a Row anymore
		
		if (childArea == null) {
			// the respective minimum constraint needs to use the new tab
			minContentHeight.setLeftSide(-1, top, 1, bottom);

			// if a maximum constraint was set, then it nees to use the new tab
			if (maxContentHeight != null)
				maxContentHeight.setLeftSide(-1, top, 1, bottom);
		} else
			updateVertical(); // the constraints between outer area and childArea need to use the new tab
		ls.invalidateLayout();
	}

	/**
	 * The row that defines the top and bottom tabs. May be null.
	 */
	public Row getRow() { return row; }

	public void setRow(Row value) throws Exception {
		setTop(value.getTop());
		setBottom(value.getBottom());
		row = value;
		ls.invalidateLayout();
	}

	/**
	 * The column that defines the left and right tabs. May be null.
	 */
	public Column getColumn() { return column; }

	public void setColumn(Column value) throws Exception {
		setLeft(value.getLeft());
		setRight(value.getRight());
		column = value;
		ls.invalidateLayout();
	}

	/**
	 * The control that is the content of the area.
	 */
	public JComponent getContent() {
		// if we are using a childArea then it contains the control,
		// otherwise this area contains the control directly
		return (childArea == null) ? content : childArea.getContent(); 
	}

	public void setContent(JComponent value) {
		if (childArea == null) content = value;
		else childArea.content = value;
		ls.invalidateLayout();
	}

	/**
	 * Left tab of the area's content. May be different from the left tab of the area.
	 */
	public XTab getContentLeft() {
		return (childArea == null) ? left : childArea.left;
	}

	/**
	 * Top tab of the area's content. May be different from the top tab of the area.
	 */
	public YTab getContentTop() {
		return (childArea == null) ? top : childArea.top;
	}

	/**
	 * Right tab of the area's content. May be different from the right tab of the area.
	 */
	public XTab getContentRight() {
		return (childArea == null) ? right : childArea.right;
	}

	/**
	 * Bottom tab of the area's content. May be different from the bottom tab of the area.
	 */
	public YTab getContentBottom() {
		return (childArea == null) ? bottom : childArea.bottom;
	}

	/**
	 * Minimum size of the area's content. May be different from the minimum size of the area.
	 */
	public Dimension getMinContentSize() {
		return (childArea == null) ? minContentSize : childArea.minContentSize;
	}

	public void setMinContentSize(Dimension value) throws Exception {
		if (childArea == null) {
			minContentSize = value;
			minContentWidth.setRightSide(minContentSize.getWidth());
			minContentHeight.setRightSide(minContentSize.getHeight());
		}
		else childArea.setMinContentSize(value);
		ls.invalidateLayout();
	}

	/**
	 * Maximum size of the area's content. May be different from the maximum size of the area.
	 */
	public Dimension getMaxContentSize() {
		return (childArea == null) ? maxContentSize : childArea.maxContentSize;
	}

	public void setMaxContentSize(Dimension value) throws Exception {
		if (childArea == null) {
			maxContentSize = value;
			if (maxContentWidth == null) { // no max constraints set yet
				maxContentWidth = ls.addConstraint(-1, left, 1, right,
						OperatorType.LE, maxContentSize.getWidth());
				constraints.add(maxContentWidth);
				maxContentHeight = ls.addConstraint(-1, top, 1, bottom,
						OperatorType.LE, maxContentSize.getHeight());
				constraints.add(maxContentHeight);
			}
			else {
				maxContentWidth.setRightSide(maxContentSize.getWidth());
				maxContentHeight.setRightSide(maxContentSize.getHeight());
			}
		}
		else childArea.setMaxContentSize(value);
		ls.invalidateLayout();
	}

	/**
	 * Preferred size of the area's content. May be different from the preferred size of the area.
	 * Manual changes of PrefContentSize are ignored unless autoPrefContentSize is set to false.
	 */
	public Dimension getPrefContentSize() {
		return (childArea == null) ? prefContentSize : childArea.prefContentSize;
	}

	public void setPrefContentSize(Dimension value) throws LpSolveException, Exception {
		if (childArea == null) {
			prefContentSize = value;
			if (prefContentWidth == null) { // no pref constraints set yet
				prefContentWidth = ls.addConstraint(-1, left, 1, right,
						OperatorType.EQ, prefContentSize.getWidth(), shrinkRigidity.getWidth(), expandRigidity.getWidth());
				constraints.add(prefContentWidth);
				prefContentHeight = ls.addConstraint(-1, top, 1, bottom,
						OperatorType.EQ, prefContentSize.getHeight(), shrinkRigidity.getHeight(), expandRigidity.getHeight());
				constraints.add(prefContentHeight);
			}
			else {
				prefContentWidth.setRightSide(value.getWidth());
				prefContentHeight.setRightSide(value.getHeight());
			}
		}
		else childArea.setPrefContentSize(value);
		ls.invalidateLayout();
	}

	/**
	 * The reluctance with which the area's content shrinks below its preferred size.
	 * The bigger the less likely is such shrinking.
	 */
	public Dimension getShrinkRigidity() {
		return (childArea == null) ? shrinkRigidity : childArea.shrinkRigidity;
	}

	public void setShrinkRigidity(Dimension value) throws Exception {
		if (childArea == null) {
			shrinkRigidity = value;
			if (prefContentWidth != null) { // rigidities are only relevant if a preferred size is set
				prefContentWidth.setPenaltyNeg(value.getWidth());
				prefContentHeight.setPenaltyNeg(value.getHeight());
			}
		}
		else childArea.setShrinkRigidity(value);
		ls.invalidateLayout();
	}

	/**
	 * The reluctance with which the area's content expands over its preferred size.
	 * The bigger the less likely is such expansion.
	 */
	public Dimension getExpandRigidity() {
		return (childArea == null) ? expandRigidity : childArea.expandRigidity;
	}

	public void setExpandRigidity(Dimension value) throws Exception {
		if (childArea == null) {
			expandRigidity = value;
			if (prefContentWidth != null) { // rigidities are only relevant if a preferred size is set
				prefContentWidth.setPenaltyPos(value.getWidth());
				prefContentHeight.setPenaltyPos(value.getHeight());
			}
		}
		else childArea.setExpandRigidity(value);
		ls.invalidateLayout();
	}

	/**
	 * Aspect ratio of the area's content. May be different from the aspect ratio of the area.
	 */
	public double getContentAspectRatio() {
		return (childArea == null) ? contentAspectRatio : childArea.contentAspectRatio;
	}

	public void setContentAspectRatio(double value) throws Exception {
		if (childArea == null) {
			contentAspectRatio = value;
			if (contentAspectRatioC == null) { // no aspect ratio constraint set yet
				contentAspectRatioC = ls.addConstraint(
						-1, left, 1, right, value, top, -value, bottom,
						OperatorType.EQ, 0);
				constraints.add(contentAspectRatioC);
			}
			else {
				contentAspectRatioC.setLeftSide(
						-1, left, 1, right, value, top, -value, bottom);
			}
		}
		else childArea.setContentAspectRatio(value);
		ls.invalidateLayout();
	}

	/**
	 * Horizontal alignment of the content in its area.
	 */
	public HorizontalAlignment getHAlignment() { return hAlignment; }

	public void setHAlignment(HorizontalAlignment value) throws Exception {
		hAlignment = value;
		updateHorizontal();
		ls.invalidateLayout();
	}

	/**
	 * Vertical alignment of the content in its area.
	 */
	public VerticalAlignment getVAlignment() { return vAlignment; }

	public void setVAlignment(VerticalAlignment value) throws Exception {
		vAlignment = value;
		updateVertical();
		ls.invalidateLayout();
	}

	/**
	 * Left margin between area and its content.
	 */
	public int getLeftMargin() { return leftMargin; }

	public void setLeftMargin(int value) throws Exception {
		leftMargin = value;
		updateHorizontal();
		ls.invalidateLayout();
	}

	/**
	 * Top margin between area and its content.
	 */
	public int getTopMargin() { return topMargin; }

	public void setTopMargin(int value) throws Exception {
		topMargin = value;
		updateVertical();
		ls.invalidateLayout();
	}

	/**
	 * Right margin between area and its content.
	 */
	public int getRightMargin() { return rightMargin; }

	public void setRightMargin(int value) throws Exception {
		rightMargin = value;
		updateHorizontal();
		ls.invalidateLayout();
	}

	/**
	 * Bottom margin between area and its content.
	 */
	public int getBottomMargin() { return bottomMargin; }

	public void setBottomMargin(int value) throws Exception {
		bottomMargin = value;
		updateVertical();
		ls.invalidateLayout();
	}

	/**
	 * Adds a childArea to this area, together with constraints that specify the relative location of the 
	 * childArea within this area. It is called when such a childArea becomes necessary, 
	 * i.e. when the user requests margins or special alignment.
	 */
	void initChildArea() throws Exception {
		// add a child area with new tabs,
		// and add constraints that set its tabs to be equal to the 
		// coresponding tabs of this area (for a start)
		childArea = new Area(ls, new XTab(ls), new YTab(ls), new XTab(ls), new YTab(ls), content, new Dimension(0, 0));
		leftConstraint = getLeft().isEqual(childArea.getLeft());
		constraints.add(leftConstraint);
		topConstraint = getTop().isEqual(childArea.getTop());
		constraints.add(topConstraint);
		rightConstraint = childArea.getRight().isEqual(getRight());
		constraints.add(rightConstraint);
		bottomConstraint = childArea.getBottom().isEqual(getBottom());
		constraints.add(bottomConstraint);

		// remove the minimum content size constraints from this area
		// and copy the minimum content size setting to the childArea
		constraints.remove(minContentWidth);
		minContentWidth.remove();
		minContentWidth = childArea.minContentWidth;
		constraints.remove(minContentHeight);
		minContentHeight.remove();
		minContentHeight = childArea.minContentHeight;
		childArea.setMinContentSize(minContentSize);

		// if there are maximum content size constraints on this area, 
		// change them so that they refer to the tabs of the childArea 
		// and copy the minimum content size settings to the childArea
		if (maxContentWidth != null) {
			childArea.maxContentSize = maxContentSize;
			childArea.maxContentWidth = maxContentWidth;
			maxContentWidth.setLeftSide(-1, childArea.getLeft(),
					1, childArea.getRight());
			childArea.maxContentHeight = maxContentHeight;
			maxContentHeight.setLeftSide(-1, childArea.getTop(),
					1, childArea.getBottom());
		}

		// if there are preferred content size constraints on this area, 
		// change them so that they refer to the tabs of the childArea 
		// and copy the preferred content size settings to the childArea
		if (prefContentHeight != null) {
			childArea.prefContentSize = prefContentSize;
			childArea.shrinkRigidity = shrinkRigidity;
			childArea.expandRigidity = expandRigidity;
			childArea.prefContentWidth = prefContentWidth;
			prefContentWidth.setLeftSide(-1, childArea.getLeft(),
					1, childArea.getRight());
			childArea.prefContentHeight = prefContentHeight;
			prefContentHeight.setLeftSide(-1, childArea.getTop(),
					1, childArea.getBottom());
		}
	}

	/**
	 * Update the constraints for horizontal margins and alignment.
	 */
	void updateHorizontal() throws Exception {
		// if the area does not have a childAdrea yet, this is the time to add it
		if (childArea == null) initChildArea();

		// change the constraints leftConstraint and rightConstraint so that the horizontal alignment 
		// and margins of the childArea within this area are as specified by the user
		if (hAlignment == HorizontalAlignment.LEFT) {
			leftConstraint.setLeftSide(-1, getLeft(), 1, childArea.getLeft());
			leftConstraint.setOp(OperatorType.EQ);
			leftConstraint.setRightSide(leftMargin);
			rightConstraint.setLeftSide(-1, childArea.getRight(), 1, getRight());
			rightConstraint.setOp(OperatorType.GE);
			rightConstraint.setRightSide(rightMargin);
		}
		else if (hAlignment == HorizontalAlignment.RIGHT) {
			leftConstraint.setLeftSide(-1, getLeft(), 1, childArea.getLeft());
			leftConstraint.setOp(OperatorType.GE);
			leftConstraint.setRightSide(leftMargin);
			rightConstraint.setLeftSide(-1, childArea.getRight(), 1, getRight());
			rightConstraint.setOp(OperatorType.EQ);
			rightConstraint.setRightSide(rightMargin);
		}
		else if (hAlignment == HorizontalAlignment.CENTER) {
			leftConstraint.setLeftSide(-1, getLeft(), 1, childArea.getLeft());
			leftConstraint.setOp(OperatorType.GE);
			leftConstraint.setRightSide(Math.max(leftMargin, rightMargin));
			rightConstraint.setLeftSide(-1, getLeft(), 1, childArea.getLeft(),
					1, childArea.getRight(), -1, getRight());
			rightConstraint.setOp(OperatorType.EQ);
			rightConstraint.setRightSide(0);
		}
		else if (hAlignment == HorizontalAlignment.FILL) {
			leftConstraint.setLeftSide(-1, getLeft(), 1, childArea.getLeft());
			leftConstraint.setOp(OperatorType.EQ);
			leftConstraint.setRightSide(leftMargin);
			rightConstraint.setLeftSide(-1, childArea.getRight(), 1, getRight());
			rightConstraint.setOp(OperatorType.EQ);
			rightConstraint.setRightSide(rightMargin);
		}
		else if (hAlignment == HorizontalAlignment.NONE) {
			leftConstraint.setLeftSide(-1, getLeft(), 1, childArea.getLeft());
			leftConstraint.setOp(OperatorType.GE);
			leftConstraint.setRightSide(leftMargin);
			rightConstraint.setLeftSide(-1, childArea.getRight(), 1, getRight());
			rightConstraint.setOp(OperatorType.GE);
			rightConstraint.setRightSide(rightMargin);
		}
	}

	/**
	 * Update the constraints for vertical margins and alignment.
	 */
	void updateVertical() throws Exception {
		// if the area does not have a childAdrea yet, this is the time to add it
		if (childArea == null) initChildArea();

		// change the constraints topConstraint and bottomConstraint so that the vertical alignment 
		// and margins of the childArea within this area are as specified by the user
		if (vAlignment == VerticalAlignment.TOP) {
			topConstraint.setLeftSide(-1, getTop(), 1, childArea.getTop());
			topConstraint.setOp(OperatorType.EQ);
			topConstraint.setRightSide(topMargin);
			bottomConstraint.setLeftSide(-1, childArea.getBottom(), 1, getBottom());
			bottomConstraint.setOp(OperatorType.GE);
			bottomConstraint.setRightSide(bottomMargin);
		}
		else if (vAlignment == VerticalAlignment.BOTTOM) {
			topConstraint.setLeftSide(-1, getTop(), 1, childArea.getTop());
			topConstraint.setOp(OperatorType.GE);
			topConstraint.setRightSide(topMargin);
			bottomConstraint.setLeftSide(-1, childArea.getBottom(), 1, getBottom());
			bottomConstraint.setOp(OperatorType.EQ);
			bottomConstraint.setRightSide(bottomMargin);
		}
		else if (vAlignment == VerticalAlignment.CENTER) {
			topConstraint.setLeftSide(-1, getTop(), 1, childArea.getTop());
			topConstraint.setOp(OperatorType.GE);
			topConstraint.setRightSide(Math.max(topMargin, bottomMargin));
			bottomConstraint.setLeftSide(-1, getTop(), 1, childArea.getTop(), 
					1, childArea.getBottom(), -1, getBottom());
			bottomConstraint.setOp(OperatorType.EQ);
			bottomConstraint.setRightSide(0);
		}
		else if (vAlignment == VerticalAlignment.FILL) {
			topConstraint.setLeftSide(-1, getTop(), 1, childArea.getTop());
			topConstraint.setOp(OperatorType.EQ);
			topConstraint.setRightSide(topMargin);
			bottomConstraint.setLeftSide(-1, childArea.getBottom(), 1, getBottom());
			bottomConstraint.setOp(OperatorType.EQ);
			bottomConstraint.setRightSide(bottomMargin);
		}
		else if (vAlignment == VerticalAlignment.NONE) {
			topConstraint.setLeftSide(-1, getTop(), 1, childArea.getTop());
			topConstraint.setOp(OperatorType.GE);
			topConstraint.setRightSide(topMargin);
			bottomConstraint.setLeftSide(-1, childArea.getBottom(), 1, getBottom());
			bottomConstraint.setOp(OperatorType.GE);
			bottomConstraint.setRightSide(bottomMargin);
		}
	}

	/**
	 * Sets the preferred size according to the content's PreferredSize method, 
	 * and the rigidities according to heuristics.
	 */
	public void setDefaultPrefContentSize() throws LpSolveException, Exception {
		if (getContent() == null) {
			setPrefContentSize(new Dimension(0, 0));
			setShrinkRigidity(new Dimension(0, 0));
			setExpandRigidity(new Dimension(0, 0));
			return;
		}
		
		if (getPrefContentSize() != getContent().getPreferredSize()) {
			setPrefContentSize(getContent().getPreferredSize());
			ls.invalidateLayout();
		}
		

		/* use heuristics for rigidities:
		 * controls with naturally constant content size 
		 * are less likely to change their size */
		if (getContent() instanceof JButton
				|| getContent() instanceof JRadioButton
				|| getContent() instanceof JCheckBox
				|| getContent() instanceof JLabel
				//|| getContent() instanceof LinkLabel
				//|| getContent() instanceof PictureBox
				|| getContent() instanceof JProgressBar) {
			//|| getContent() instanceof NumericUpDown) {
			setShrinkRigidity(new Dimension(4, 4));
			setExpandRigidity(new Dimension(3, 3));
		}
		else {
			setShrinkRigidity(new Dimension(2, 2));
			setExpandRigidity(new Dimension(1, 1));
		}
	}

	void doLayout() {
		if(getContent() == null) return; // empty areas need no layout

		// if there is a childArea, then it is the childArea that actually contains the content
		Area a = (childArea != null) ? childArea : this;

		// set content location and size
		a.getContent().setLocation((new Point((int)Math.round(a.getLeft().value), (int)Math.round(a.getTop().value))));
		int width = (int)Math.round(a.getRight().value - a.getLeft().value);
		int height = (int)Math.round(a.getBottom().value - a.getTop().value);
		a.getContent().setSize(width, height);
	}

	public String toString() {
		return "Area(" + left.toString() + "," + top.toString() + ","
		+ right.toString() + "," + bottom.toString() + ")";
	}

	/**
	 * Sets the width of the area to be the same as the width of the given area.
	 * 
	 * @param a	the area that should have the same width
	 * @return the same-width constraint
	 */
	public Constraint hasSameWidthAs(Area a) throws Exception {
		return ls.addConstraint(-1, left, 1, right, 1, a.left, -1, a.right,
				OperatorType.EQ, 0);
	}

	/**
	 * Sets the height of the area to be the same as the height of the given area.
	 * 
	 * @param a	the area that should have the same height
	 * @return the same-height constraint
	 */
	public Constraint hasSameHeightAs(Area a) throws Exception {
		return ls.addConstraint(-1, top, 1, bottom, 1, a.top, -1, a.bottom,
				OperatorType.EQ, 0);
	}

	/**
	 * Sets the size of the area to be the same as the size of the given area.
	 * 
	 * @param a	the area that should have the same size
	 * @return a list containing a same-width and same-height constraint
	 */
	public List<Constraint> hasSameSizetAs(Area a) throws Exception {
		List<Constraint> l = new ArrayList<Constraint>();
		l.add(this.hasSameWidthAs(a));
		l.add(this.hasSameHeightAs(a));
		return l;
	}

	Area(LayoutSpec ls, XTab left, YTab top, XTab right, YTab bottom, JComponent content,
			Dimension minContentSize) throws Exception {
		this.ls = ls;
		this.left = left;
		this.right = right;
		this.top = top;
		this.bottom = bottom;
		setContent(content);
		this.minContentSize = minContentSize;

		// adds the two essential constraints of the area that make sure that the left x-tab is really
		// to the left of the right x-tab, and the top y-tab really above the bottom y-tab
		minContentWidth = ls.addConstraint(-1, left, 1, right,
				OperatorType.GE, minContentSize.getWidth());
		constraints.add(minContentWidth);
		minContentHeight = ls.addConstraint(-1, top, 1, bottom,
				OperatorType.GE, minContentSize.getHeight());
		constraints.add(minContentHeight);
	}

	Area(LayoutSpec ls, Row row, Column column, JComponent content, Dimension minContentSize) throws Exception {
		this(ls, column.getLeft(), row.getTop(), column.getRight(), row.getBottom(), content, minContentSize);
		this.row = row;
		this.column = column;
	}

	/**
	 * Removes the area from its specification.
	 */
	public void remove() throws Exception {
		if(childArea != null) childArea.remove();
		for (Constraint c : constraints) c.remove();
		ls.areas.remove(this);
	}
}
