package alm;

import java.io.*;
import java.util.Iterator;
import java.util.TreeMap;
import javax.swing.*;
import java.awt.*;

import linearProgramming.*;

/**
 * A GUI layout engine using the ALM.
 */
public class ALMEngine implements LayoutManager {

	/**
	 * The specification used for calculating the layout.
	 */
	public LayoutSpec layoutSpec;

	/**
	 * The manner in which the GUI is dynamically adjusted.
	 * The default is to fit the child controls into their parent.
	 */
	public LayoutStyleType LayoutStyle = LayoutStyleType.FIT_TO_SIZE;

	boolean activated = true;

	/**
	 * Creates a new layout engine with an empty specification.
	 */
	public ALMEngine() {
		super();
	}

	/**
	 * Creates a new layout engine with the given specification.
	 * @param ls	the layout specifiaction
	 */
	public ALMEngine(LayoutSpec ls) {
		super();
		layoutSpec = ls;
	}

//	public boolean layout(object container, LayoutEventArgs layoutEventArgs) {
//	JComponent parent = container as JComponent;
//	layout(parent);
//	return false;  // False means that container's parent should not perform layout.
//	}

	/**
	 * Calculate and set the layout.
	 * If no layout specification is given, a specification is reverse engineered automatically.
	 * @param parent	the parent control of the controls in the layout
	 */
	public void layout(Container parent) throws Exception {
		// make sure that layout events occuring during layout are ignored
		// i.e. activated is set to false during layout caluclation
		if (!activated) return;
		activated = false;

		// reverse engineer a layout specification if none was given
		if (layoutSpec == null) recoverLayout(parent);

		// if the layout engine is set to fit the GUI to the given size,
		// then the given size is enforced by setting absolute positions for Right and Bottom
		if (LayoutStyle == LayoutStyleType.FIT_TO_SIZE) {    
			layoutSpec.right.setRange(parent.getBounds().getWidth(), parent.getBounds().getWidth());  
			layoutSpec.bottom.setRange(parent.getBounds().getHeight(), parent.getBounds().getHeight());
		}

		layoutSpec.solveLayout();
		

		// if new layout is infasible, use previous layout
		if (layoutSpec.result == ResultType.INFEASIBLE) {
			activated = true; // now layout calculation is allowed to run again
			return;
		}
		if (layoutSpec.result != ResultType.OPTIMAL) {
			layoutSpec.save("failed-layout.txt");
			throw new Exception("Could not solve the layout specification ("
					+ layoutSpec.result.toString() + "). Saved specification in file failed-layout.txt");
		}

		//layoutSpec.Save("last-feasible-layout.txt");

		// change the size of the GUI according to the calculated size 
		// if the layout engine was configured to do so
		if (LayoutStyle == LayoutStyleType.ADJUST_SIZE) {
			parent.setSize(new Dimension(
					(int)Math.round(layoutSpec.right.value - layoutSpec.left.value),
					(int)Math.round(layoutSpec.bottom.value - layoutSpec.top.value)));
		}

		// set the calculated positions and sizes for every area
		for (Area a : layoutSpec.areas) a.doLayout(); 

		activated = true; // now layout calculation is allowed to run again
	}

	// #region RE
	/**
	 * Reverse engineers a GUI and recovers an ALM specification.
	 * @param parent	the parent container of the GUI
	 */
	public void recoverLayout(Container parent) throws Exception {
		layoutSpec = new LayoutSpec();
		TreeMap<Integer, XTab> xtabs = new TreeMap<Integer, XTab>();
		TreeMap<Integer, YTab> ytabs = new TreeMap<Integer, YTab>();

		xtabs.put((int)parent.getBounds().getX(), layoutSpec.left);
		xtabs.put((int)(parent.getBounds().getX() + parent.getBounds().getWidth()), layoutSpec.right);
		ytabs.put((int)parent.getBounds().getY(), layoutSpec.top);
		ytabs.put((int)(parent.getBounds().getY() + parent.getBounds().getHeight()), layoutSpec.bottom);

		for (Component c : parent.getComponents()) {
			JComponent cJ = (JComponent)c;
			XTab x1, x2;
			YTab y1, y2;

			if (xtabs.containsKey((int)cJ.getVisibleRect().getX())) 
				x1 = xtabs.get((int)cJ.getVisibleRect().getX());
			else xtabs.put((int)cJ.getVisibleRect().getX(), x1 = new XTab(layoutSpec));
			if (xtabs.containsKey((int)(cJ.getVisibleRect().getX() + cJ.getVisibleRect().getWidth())))
				x2 = xtabs.get((int)(cJ.getVisibleRect().getX() + cJ.getVisibleRect().getWidth()));
			else xtabs.put((int)(cJ.getVisibleRect().getX() + cJ.getVisibleRect().getWidth()), x2 = new XTab(layoutSpec));
			if (ytabs.containsKey((int)cJ.getVisibleRect().getY())) 
				y1 = ytabs.get((int)cJ.getVisibleRect().getY());
			else ytabs.put((int)cJ.getVisibleRect().getY(), y1 = new YTab(layoutSpec));
			if (ytabs.containsKey((int)(cJ.getVisibleRect().getY() + cJ.getVisibleRect().getHeight()))) 
				y2 = ytabs.get((int)(cJ.getVisibleRect().getY() + cJ.getVisibleRect().getHeight()));
			else ytabs.put((int)(cJ.getVisibleRect().getY() + cJ.getVisibleRect().getHeight()), y2 = new YTab(layoutSpec));

			layoutSpec.addArea(x1, y1, x2, y2, cJ);

			x2.leftLink = true;
			y2.topLink = true;
		}

		// adding additional constraints (links in the PO) for margins between areas
		XTab currentXTab = null;
		XTab previousXTab = null;
		int currentXKey = 0;
		int previousXKey = 0;
		Iterator xTabsIterator = xtabs.values().iterator();
		Iterator xKeyIterator = xtabs.keySet().iterator();
		while (xTabsIterator.hasNext()) {
			currentXTab = (XTab)xTabsIterator.next();
			currentXKey = Integer.parseInt(xKeyIterator.next().toString());
			if (!currentXTab.leftLink)
				layoutSpec.addConstraint(-1, previousXTab, 1, currentXTab,
						OperatorType.EQ, -previousXKey + currentXKey);
			previousXTab = currentXTab;
			previousXKey = currentXKey;
		}

		YTab currentYTab = null;
		YTab previousYTab = null;
		int currentYKey = 0;
		int previousYKey = 0;
		Iterator yTabsIterator = ytabs.values().iterator();
		Iterator yKeyIterator = ytabs.keySet().iterator();
		while (yTabsIterator.hasNext()) {
			currentYTab = (YTab)yTabsIterator.next();
			currentYKey = Integer.parseInt(yKeyIterator.next().toString());
			if (!currentYTab.topLink)
				layoutSpec.addConstraint(-1, previousYTab, 1, currentYTab,
						OperatorType.EQ, -previousYKey + currentYKey);
			previousYTab = currentYTab;
			previousYKey = currentYKey;
		}
	}
//	#endregion RE

	public void removeLayoutComponent(Component comp) {}

	public void layoutContainer(Container parent) {
		try {
			layout(parent);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	public void addLayoutComponent(String name, Component comp) {}
	public Dimension minimumLayoutSize(Container parent) { return parent.getSize(); }
	public Dimension preferredLayoutSize(Container parent) { return parent.getSize(); }
}