/***********************************************************
 * Code reused by Peter McClymont
 * by permission of Author: Dr. S. Manoharan
 ***********************************************************/

#include <string.h>
#include <stdio.h>

#include "GetLongOpt.h"

GetLongOpt::GetLongOpt()
{
   argV = 0;
   argC = 0;
   table = last = 0;
   ustring = "[valid options and arguments]";
   enrollDone = 0;
   enrolledPublic = 0;
   optmarker = '-';
   dirmarker = '/';
   opt_index = 1;
}

GetLongOpt&
GetLongOpt::operator=(const GetLongOpt& g)
{
   if ( this != &g ) {  // not self
      cleanup(); copy(g);
   }
   return *this;
}

void
GetLongOpt::copy(const GetLongOpt& g)
{
   // copy all the invariants
   argV = g.argV;
   ustring = g.ustring;
   enrollDone = g.enrollDone;
   enrolledPublic = g.enrolledPublic;
   optmarker = g.optmarker;
   opt_index = g.opt_index;
   table = last = 0;

   // now copy the table
   Cell *t;
   for ( t = g.table; t != 0; t = t->next ) {
      Cell *c = new Cell;
      c->option = t->option;
      c->type = t->type;
      c->description = t->description;
      c->value = t->value;
      c->status = t->status;
      c->next = 0;

      if ( last == 0 ) {
        table = last = c;
      }
      else {
        last->next = c;
        last = c;
      }
   }
}

void
GetLongOpt::cleanup()
{
   Cell *t = table;

   while ( t ) {
      Cell *tmp = t;
      t = t->next;
      delete tmp;
   }
   table = last = 0;
}

char *
GetLongOpt::basename(char * const pathname) const
{
   char *s;

   s = strrchr(pathname, dirmarker);
   if ( s == 0 ) s = pathname;
   else ++s;

   return s;
}

const char *
GetLongOpt::getarg(const int i) const
{
   if ( argV == 0 || argCount() <= 0 ) return 0;
   if ( i + opt_index >= argC ) return 0;
   if ( i < 0 ) return 0;
   return argV[i + opt_index];
}

bool
GetLongOpt::enroll(const char * const opt, const OptType t,
const char * const desc, const char * const val, const OptStatus stat)
{
   if ( enrollDone ) return false;

   Cell *c = new Cell;
   c->option = opt;
   c->type = t;
   c->description = desc ? desc : "no description available";
   c->next = 0;
   c->status = stat;

   if ( c->status == Public ) ++enrolledPublic;

   switch ( c->type ) {
   case FlagOption :
      c->value = 0;
      break;
   case ValueOption :
      c->value = val;
      break;
   }

   if ( last == 0 ) {
      table = last = c;
   }
   else {
      last->next = c;
      last = c;
   }

   return true;
}

const char *
GetLongOpt::retrieve(const char * const opt) const
{
   Cell *t;
   for ( t = table; t != 0; t = t->next ) {
      if ( strcmp(opt, t->option) == 0 )
        return t->value;
   }
   cerr << "GetLongOpt::retrieve - unenrolled option ";
   cerr << optmarker << opt << endl;
   return 0;
}

bool
GetLongOpt::parse(int argc, char **argv, const ParseAction action)
{
   pname = basename(*argv);
   argC = argc, argV = argv;
   if ( action != IgnoreUnknown) enrollDone = 1;
   if ( argc-- <= 1 ) return true;
   opt_index = 1;

   while ( argc >= 1 ) {
      char *token = *++argv; --argc;

      if ( token[0] != optmarker ) break;      // end of options
      else if ( token[0] == optmarker && token[1] == optmarker ) {
        ++opt_index;   // point to the next argument
        break;
      } // end of options too ...
      else if ( token[0] == optmarker && token[1] == '\0' ) {
        ++opt_index;   // point to the next argument
        break;
      } // end of options again?

      ++opt_index;
      char *tmptoken = ++token;
      while ( *tmptoken && *tmptoken != '=' )
        ++tmptoken;
      const int optionLength = tmptoken - token;

      Cell *t;
      enum { NoMatch, MatchFound } matchStatus = NoMatch;
      Cell *match = 0; // pointer to the (partially) matched cell
      for ( t = table; t != 0; t = t->next ) {
        if ( strncmp(t->option, token, optionLength) == 0 ) {
           if ( strlen(t->option) == (size_t)(tmptoken - token) ) {
              // An exact match found. Break loop.
              matchStatus = MatchFound;
              match = t;
              break;
           }
           else {
              // A possible partial match found.
              // Protected options are not allowed to partially match.
              if ( t->status != Protected && matchStatus == NoMatch ) {
                 matchStatus = MatchFound;
                 match = t;    // Partial match confirmed.
                 // Don't break here, there may be an exact match
                 // coming later on.
              }
           }
        }
      }

      if ( matchStatus == MatchFound && match->status != Private ) {
        SCstatus stat = setcell(match, tmptoken, *(argv+1), pname);
        if ( stat == SC_ERROR ) return false;
        else if ( stat == 1 ) {
           ++argv; --argc; ++opt_index;
        }
      }
      else {
        // we have seen an unknown option here
        if ( action != IgnoreUnknown ) {
           cerr << pname << ": unrecognized option ";
           cerr << optmarker << strtok(token,"= ") << endl;
           return false;               // no match
        }
      }

   } // end while

   return true;
}

bool
GetLongOpt::parse(char * const str, char * const p,
const ParseAction action)
{
   if ( action != IgnoreUnknown) enrollDone = 1;
   char *token = strtok(str, " \t");
   
   const char * name = p ? p :"GetLongOpt";
   //char *name = "GetLongOpt";

   while ( token ) {
      if ( token[0] != optmarker
           || (token[0] == optmarker && token[1] == optmarker)
           || (token[0] == optmarker && token[1] == '\0') ) {
        cerr << name << ": nonoptions not allowed" << endl;
        return false;  // end of options
      }

      char *ladtoken = 0;      // lookahead token
      char *tmptoken = ++token;
      while ( *tmptoken && *tmptoken != '=' )
        ++tmptoken;
      const int optionLength = tmptoken - token;

      Cell *t;
      enum { NoMatch, MatchFound } matchStatus = NoMatch;
      Cell *match = 0; // pointer to the (partially) matched cell
      for ( t = table; t != 0; t = t->next ) {
        if ( strncmp(t->option, token, optionLength) == 0 ) {
           if ( strlen(t->option) == (size_t)(tmptoken - token) ) {
              // An exact match found. Break loop.
              matchStatus = MatchFound;
              match = t;
              break;
           }
           else {
              // A possible partial match found.
              // Protected options are not allowed to partially match.
              if ( t->status != Protected && matchStatus == NoMatch ) {
                 matchStatus = MatchFound;
                 match = t;    // Partial match confirmed.
                 // Don't break here, there may be an exact match
                 // coming later on.
              }
           }
        }
      }

      if ( matchStatus == MatchFound && match->status != Private ) {
        ladtoken = strtok(0, " \t");
        SCstatus stat = setcell(match, tmptoken, ladtoken, name);
        if ( stat == SC_ERROR ) return false;
        else if ( stat == 1 ) {
           ladtoken = 0;
        }
      }
      else if ( matchStatus == NoMatch ) {
        // we have seen an unknown option here
        if ( action != IgnoreUnknown ) {
           cerr << name << ": unrecognized option ";
           cerr << optmarker << strtok(token,"= ") << endl;
           return false;               // no match
        }
      }

      token = ladtoken ? ladtoken : strtok(0, " \t");
   } // end while

   return true;
}

// ----------------------------------------------------------------
// GetLongOpt::setcell returns
//    SC_ERROR          if there was an error
//    SC_TOKCONSUMED    if the nexttoken was not consumed
//    SC_TOKNOTCONSUMED if the nexttoken was consumed
// ----------------------------------------------------------------

GetLongOpt::SCstatus
GetLongOpt::setcell(Cell *c, char *valtoken, char *nexttoken,
const char *name)
{
   if ( c == 0 ) return SC_ERROR;

   switch ( c->type ) {
   case FlagOption :
      if ( *valtoken == '=' ) {
        cerr << name << ": unsolicited value for flag ";
        cerr << optmarker << c->option << endl;
        return SC_ERROR;       // unsolicited value specification
      }
      c->value = (char *) ~0;
      return SC_TOKCONSUMED;
   case ValueOption :
      if ( *valtoken == '=' ) {
        c->value = ++valtoken;
        return SC_TOKCONSUMED;
      }
      else {
        if ( nexttoken != 0 && nexttoken[0] != optmarker ) {
           c->value = nexttoken;
           return SC_TOKNOTCONSUMED;
        }
        else {
           cerr << name << ": mandatory value for ";
           cerr << optmarker << c->option << " not specified" << endl;
           return SC_ERROR;    // mandatory value not specified
        }
      }
   default :
      break;
   }
   return SC_ERROR;
}

void
GetLongOpt::usage(ostream &outfile) const
{
   Cell *t;

   outfile << "usage: " << pname << " " << ustring << "\n";
   for ( t = table; t != 0; t = t->next ) {
      if ( t->status == Public ) {
        outfile << "\t" << optmarker << t->option;
        if ( t->type == ValueOption )
           outfile << " <$val>";
        outfile << " (" << t->description << ")\n";
      }
   }
   outfile.flush();
}

void
GetLongOpt::manpage(ostream &outfile) const
{
   Cell *t;

   for ( t = table; t != 0; t = t->next ) {
      if ( t->status == Public ) {
        outfile << ".TP\n.BI \\" << optmarker << t->option;
        if ( t->type == ValueOption )
           outfile << " \" $val \"";
        char c = *(t->description);
        if ( c <= 'z' && c >= 'a' ) c += 'A' - 'a';
        outfile << "\n" << c << (t->description + 1) << ".\n";
      }
   }
   outfile.flush();
}

void
GetLongOpt::valuesP(ostream &outfile) const
{
   Cell *t;

   for ( t = table; t != 0; t = t->next ) {
      if ( t->status == Public ) {
        if ( t->type == ValueOption ) {
           outfile << "\t" << t->option << "\t" << t->value << "\n";
        }
        else if ( t->type == FlagOption ) {
           outfile << "\t" << t->option << "\t"
                   << (t->value ? "ON" : "OFF") << "\n";
        }
      }
   }
   outfile.flush();
}

void
GetLongOpt::values(ostream &outfile) const
{
   Cell *t1, *t2;      // t1 and t2 point to columns 1 and 2 resp.
   int k;              // loop index
   long prev_flags = outfile.flags();
   char prev_fill = outfile.fill('.');

   int rows = (enrolledPublic >> 1) + (enrolledPublic & 0x1);
   int t2offset = rows;                // offset for t2 from the cell table
   for ( t2 = table; t2offset > 0; t2 = t2->next) {
      if ( t2->status == Public ) {
        --t2offset;
      }
   }
   // t2 now holds the first cell for the 2nd column ...
   t1 = table; // t1 holds the first cell for the 1st column ...

   for ( k = 0; k < rows; ++k ) {
      // first get to the next Public option for column 1 ...
      while ( t1 && t1->status != Public )
        t1 = t1->next;
      // print it ...
      if ( t1->status == Public ) {
        if ( t1->type == ValueOption ) {
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << " ";
           outfile.width(23);
           outfile.flags(prev_flags | ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << t1->option;
           outfile.width(15);
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    | ios::right);
           outfile << (t1->value ? t1->value : "OFF");
        }
        else if ( t1->type == FlagOption ) {
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << " ";
           outfile.width(23);
           outfile.flags(prev_flags | ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << t1->option;
           outfile.width(15);
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    | ios::right);
           outfile << (t1->value ? "ON" : "OFF");
        }
      }

      // first get to the next Public option for column 2 ...
      while ( t2 && t2->status != Public )
        t2 = t2->next;
      // print it ...
      if ( t2 && t2->status == Public ) {
        if ( t2->type == ValueOption ) {
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << " ";
           outfile.width(23);
           outfile.flags(prev_flags | ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << t2->option;
           outfile.width(15);
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    | ios::right);
           outfile << (t2->value ? t2->value : "OFF");
        }
        else if ( t2->type == FlagOption ) {
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << " ";
           outfile.width(23);
           outfile.flags(prev_flags | ios::left & ~ios::internal
                                    & ~ios::right);
           outfile << t2->option;
           outfile.width(15);
           outfile.flags(prev_flags & ~ios::left & ~ios::internal
                                    | ios::right);
           outfile << (t2->value ? "ON" : "OFF");
        }
      }
      t1 = t1->next; t2 = t2 ? t2->next : 0;
      outfile << endl;
   }

   outfile.flags(prev_flags);
   prev_fill = outfile.fill(prev_fill);
}

