/***********************************************************************
                          Dsim_register.cc
                          Peter McClymont
                     University of Auckland
************************************************************************/

#include "Dsim_register.h"
#include "Dsim_types.h"


/**
 * overloaded operator << for use normally with cout.
 **/
ostream&
operator<<(ostream& os, const Dsim_register&)
{
   return os;
}

/**
 * main initial constructor.
 **/
Dsim_register::Dsim_register(char *n, Dsim_data d)
{
   name = n;
   data = d;
}

Dsim_register::Dsim_register(char *n)
{
   Dsim_register(n,0);
}

/**
 * cleanup is the actual function that does the 
 * the destruction . Called by the destructor.
 **/
void
Dsim_register::cleanup()
{
}

/**
 * deep copying . 
 **/
void
Dsim_register::copy(const Dsim_register&)
{
   assert(0);
}

/**
 * For things like Dsim_register a = b;
 * overloaded operator = 
 **/
Dsim_register&
Dsim_register::operator=(const Dsim_register& c)
{
   // making sure that the passed parameter 
   // is not itself.
   if ( this != &c ) {	
      // do a cleanup and copy.	
      cleanup(); copy(c);
   }

   return *this;
}

// Implemented methods
/**
 * The general implementation of registers
 * due to the complexity involved in the
 * methods of access of registers is to
 * have one 32 bit Dsim_data type as the
 * entire representation of the register
 * and use bit-masks to play around with
 * the internals of that 32 bit data type.
 **/


/**
 * gets the entire 32 bit
 * value of the register.
 **/
Dsim_data
Dsim_register::getData()
{
   return data;
}

/**
 * gets the upper 16 bits 
 * of the register.
 * ________________________________
 * XXXXXXXXXXXXXXXX________________
 * ________________________________
 **/
Dsim_data
Dsim_register::getUpper16()
{
   return data >> 16;     
}

/**
 * gets the lower 16 bits
 * of the register
 *
 * ________________________________
 * ________________XXXXXXXXXXXXXXXX
 * ________________________________
 *
 **/
Dsim_data
Dsim_register::getLower16()
{
   return data & 0xffff ;
}

/**
 * gets the upper 8 bits of the
 * lower half of the register.
 *
 * --------------------------------
 * ________________xxxxxxxx________
 * ________________________________
 *
 **/
Dsim_data
Dsim_register::getLU8()
{
   return getLower16() >> 8;    
}

/**
 * gets the lower 8 bits of the
 * lower half of the register.
 *
 * --------------------------------
 * ________________________xxxxxxxx
 * ________________________________
 *
 **/
Dsim_data
Dsim_register::getLL8()
{
   return getLower16() & 0xff;
}


/** 
 * sets the entire 32 bit register value
 * to the passed value.
 **/
void
Dsim_register::setData(Dsim_data d)
{
   data = d;
}

/**
 * sets the upper 16 bits of the register
 * to the passed value
 **/
void
Dsim_register::setUpper16(Dsim_data d)
{
   // assuming d is always 16 bit here
   data = (data & 0xffff) | (d << 16); 
}


/**
 * sets the lower 16 bits of the register
 * to the passed value
 **/
void
Dsim_register::setLower16(Dsim_data d)
{
   //  assuming d is always 16 bit
   data = ((data >> 16) << 16) | d ;
}


/**
 * sets the upper 8 bits of the lower 16 bits
 * of the register to the passed value.
 **/
void
Dsim_register::setLU8(Dsim_data d)
{
   //assume that d is 8 bit
   //will be implemented as needed
}


/**
 * sets the lower 8 bits of the lower 16 bits
 * of the register to the passed value.
 **/
void
Dsim_register::setLL8(Dsim_data d)
{
   //assume that d is 8 bit
   data = ((data >> 8) << 8) | d;
}
