/*********************************************************************
                          Dsim_processor.cc
                          Peter McClymont
                     University of Auckland
************************************************************************/



#include <iostream.h>
#include <stdio.h>

#include "Dsim_processor.h"
#include "Dsim_register.h"
#include "Dsim_core.h"
#include "Dsim_instruction.h"
#include "Dsim_decode.h"
#include "Dsim_error.h"

/**
 * overloaded operator << for use normally with cout.
 **/
ostream&
operator<<(ostream& os, const Dsim_processor&)
{
   return os;
}

/**
 * main initial constructor.
 **/
Dsim_processor::Dsim_processor(Dsim_core *c)
{
   core = c;
   sys = new Dsim_syscall();
   decoder = new Dsim_decode();

   //initialize all the registers
   EAX = new Dsim_register("EAX");
   EBX = new Dsim_register("EBX");
   ECX = new Dsim_register("ECX");
   EDX = new Dsim_register("EDX");
   // extended base pointer
   EBP = new Dsim_register("EBP");
   // extended source index
   ESI = new Dsim_register("ESI");
   // extended data index
   EDI = new Dsim_register("EDI");
   EFLAG = new Dsim_register("EFLAG");

   // the instruction pointer  
   EIP = new Dsim_register("EIP");

   // setting the instruction pointer to the text segment
   EIP->setData(core->get_entry());
   
   // the stack pointer
   ESP = new Dsim_register("ESP");
   // setting stack pointer to the initial stack pointer
   ESP->setData(core->getInitStackValue());
   

   CS = new Dsim_register("CS");
   SS = new Dsim_register("SS");
   DS = new Dsim_register("DS");
   ES = new Dsim_register("ES");
   FS = new Dsim_register("FS");
   GS = new Dsim_register("GS");

   FPUControl = new Dsim_register("FPUControl");
   FPUControl->setData(0x600); // initial default value of FPU Control word

   start();
}


/**
 * sets the ball rolling... to start the clock
 * cycles. Perhaps should rename it to something
 * else.
 **/
void
Dsim_processor::start()
{
   int sum = 0;
   while(true) {
      cycle(&sum);
   }
}


/** 
 * Representation of what a processor does every clock
 * cycle. Quite a lot of genuine features have not 
 * been implemented yet. Currently just does what 
 * is needed for bare emulation.
 **/
void
Dsim_processor::cycle(int *sum)
{
   
   int i = 0;
   Dsim_instruction current;
   current.reset();
   int instruction[16];

   for(i = 0; i < 16; i++) {
     instruction[i] = core->inspectByte(EIP->getData() + i);
     }

   int *data = instruction;


   indent();
   cout << "0x" << EIP->getData() << " ";  
   int lendis = decoder->disasm(data, &current);

   execute(&current, lendis);

   *sum += lendis;
}




void Dsim_processor::setFlags32(int value) { // used by test
  if (value == 0) setFlag(64);
  else if (getFlagStatus(64)) clearFlag(64);
  }

bool Dsim_processor::getFlagStatus(int value) {
  return ((EFLAG->getData() & value) != 0);
  }

void Dsim_processor::setFlag(int value) {
  EFLAG->setData(EFLAG->getData() | value);
  }

void Dsim_processor::clearFlag(int value) {
  EFLAG->setData(EFLAG->getData() ^ value);  
  }

void Dsim_processor::setFlags32(long long c, long long b, long long value) {
  if (value == 0) setFlag(64);
  else if (getFlagStatus(64)) clearFlag(64);
  
  if ((value >> 32) != 0x0) setFlag(1);
  else if (getFlagStatus(1)) clearFlag(1);
  
  if ((value >> 31) == 0x1) setFlag(128);
  else if (getFlagStatus(128)) clearFlag(128);
  
  if ((value >> 32) != 0x0) setFlag(2048);
  else if (getFlagStatus(2048)) clearFlag(2048);
  
  }

int Dsim_processor::getDoubleWord (int addr) {
  return ((core->inspectByte(addr + 3) << 24) + (core->inspectByte(addr + 2) << 16) + (core->inspectByte(addr + 1) << 8) + (core->inspectByte(addr + 0)));
  }

void Dsim_processor::putDoubleWord(int addr, int word) {
  core->insertByte(addr, word & 255);
  core->insertByte(addr + 1, (word >> 8) & 255);
  core->insertByte(addr + 2, (word >> 16) & 255);
  core->insertByte(addr + 3, (word >> 24) & 255);
  }

int Dsim_processor::getWord(int addr) {
  return ((core->inspectByte(addr + 1) << 8) + (core->inspectByte(addr + 0)));
  }

void Dsim_processor::putWord(int addr, int word) {
  core->insertByte(addr, word & 255);
  core->insertByte(addr + 1, (word >> 8) & 255);
  }

int Dsim_processor::getByte(int addr) {
  return (core->inspectByte(addr + 0));
  }

void Dsim_processor::putByte(int addr, int word) {
  core->insertByte(addr, word & 255);
  }

int Dsim_processor::getSIB32(Dsim_instruction *curr_ins) {
 
  int a = curr_ins->getI(), b = 0, c = 0;
  Dsim_register* r1 = get_register(curr_ins->getB());
  b = r1->getData();  // B data.

  if (a != 0x04) {
    r1 = get_register(a);
    a = r1->getData();    // I data, 0 if the I value is 4.
    }
  else a = 0;

  if (b == 0x05 & curr_ins->getMod() == 0x3) c = curr_ins->getOffset();

  if (b == 0x05 & curr_ins->getMod() == 0x0) b = 0x0;	// exception to the rule.

  switch(curr_ins->getS()) {
    case 0x00:
      return (c + b + a);
      break;
    case 0x01:
      return (c + b + (a * 2));
      break;
    case 0x02:
      return (c + b + (a * 4));
      break;
    case 0x03:
      return (b + (a * 8));
      break;
      }
   return ~0;
   }

int Dsim_processor::getRM32Address(Dsim_instruction *curr_ins) {
  return getRM(curr_ins);
  }

int Dsim_processor::getRM(Dsim_instruction *curr_ins) {
  
  int a = 0;
  Dsim_register* r1 = get_register(curr_ins->getRm());
  int reg = r1->getData();
  switch(curr_ins->getMod()) {
    case 0x00:
      if (curr_ins->getRm() != 0x04 & curr_ins->getRm() != 0x05) {
        
        return reg;
        }
      else if (curr_ins->getRm() == 0x04) return getSIB32(curr_ins);
      else if (curr_ins->getRm() == 0x05) {
        a = curr_ins->getOffset();

        return curr_ins->getOffset();
        }
      break;
    case 0x01:
      a = curr_ins->getOffset();
      
      if (a >= 0x80) a = a + 0xffffff00; // sign extending
      if (curr_ins->getRm() != 0x04) return (reg + a);
      else return (a + getSIB32(curr_ins));
      break;
    case 0x02:
      a = curr_ins->getOffset();
      
      if (curr_ins->getRm() != 0x04) return reg + a;
      else return (a + getSIB32(curr_ins));
      break;
    case 0x03:
      return reg;
      break;
      }
    return 0;
    }


int Dsim_processor::getRM32(Dsim_instruction *curr_ins) {
  if (curr_ins->getMod() == 0x3)
    return getRM(curr_ins);
  else return getDoubleWord(getRM(curr_ins));
  return ~0;
  }
int Dsim_processor::getRM16(Dsim_instruction *curr_ins) {
  if (curr_ins->getMod() == 0x3)
    return (getRM(curr_ins) & 0xffff);
  else return getWord(getRM(curr_ins));
  return ~0;
  }
int Dsim_processor::getRM8(Dsim_instruction *curr_ins) {
  if (curr_ins->getMod() == 0x3)
    return getRMReg8(curr_ins);
  else return getByte(getRM(curr_ins));
  return ~0;
  }

void Dsim_processor::setRM32(Dsim_instruction *curr_ins, int value) {
  setRM(curr_ins, value, 32);
  }
void Dsim_processor::setRM16(Dsim_instruction *curr_ins, int value) {
  setRM(curr_ins, value, 16);
  }
void Dsim_processor::setRM8(Dsim_instruction *curr_ins, int value) {
  setRM(curr_ins, value, 8);
  }

void Dsim_processor::setRM(Dsim_instruction *curr_ins, int value, int bits) {
  
  int a = 0;
  Dsim_register* r1 = get_register(curr_ins->getRm());
  int reg = r1->getData();
  switch(curr_ins->getMod()) {
    case 0x00:
      if (curr_ins->getRm() != 0x04 & curr_ins->getRm() != 0x05) {
       
        setData(value, bits, curr_ins, curr_ins->getMod(), reg);
        }
      else if (curr_ins->getRm() == 0x05) {
        
        a = curr_ins->getOffset();
        setData(value, bits, curr_ins, curr_ins->getMod(), a);
        }
      break;
    case 0x01:
      if (curr_ins->getRm() != 0x04) {
        a = curr_ins->getOffset();
        
        if (a < 0x80) // 128
          reg = reg + a;
        else reg = reg - ((a ^ 0xff) + 1);
        setData(value, bits, curr_ins, curr_ins->getMod(), reg);
        }
      else if (curr_ins->getRm() == 0x04) {
        
        }
      break;
    case 0x02:
      if (curr_ins->getRm() != 0x04) {
        a = curr_ins->getOffset();
        
        if ((a >> 31) == 0x0)
          reg = reg + a;
        else reg = reg - ((a ^ 0xffffffff) + 1);
        setData(value, bits, curr_ins, curr_ins->getMod(), reg);
        }
      else if (curr_ins->getRm() == 0x04) {
        
        }
      break;
    case 0x03:
      setData(value, bits, curr_ins, curr_ins->getMod(), 0); 
      break;

      }
    }
  void Dsim_processor::setData(int value, int bits, Dsim_instruction *curr_ins, int mod, int reg) {
    if (mod != 0x3) {
      if (bits == 8) putByte(reg, value);
      if (bits == 16) putWord(reg, value);
      if (bits == 32) putDoubleWord(reg, value);
      }
    else {
      Dsim_register* r1 = get_register(curr_ins->getRm());
      if (bits == 8) setRMReg8(curr_ins, value);
      if (bits == 16) r1->setLower16(value);
      if (bits == 32) r1->setData(value);
      }
    }

int Dsim_processor::getRMReg8(Dsim_instruction *curr_ins) {
  switch(curr_ins->getRm()) {
    case 0x00: 
      return EAX->getLL8();
      break;
    case 0x01: 
      return ECX->getLL8();   
      break;
    case 0x02:
      return EDX->getLL8(); 
      break;
    case 0x03: 
      return EBX->getLL8();
      break;
    case 0x04:
      return EAX->getLU8();
      break;
    case 0x05: 
      return ECX->getLU8();
      break;
    case 0x06:
      return EDX->getLU8();
      break; 
    case 0x07: 
      return EBX->getLU8();
      break;
    }
  return ~0;
  }
void Dsim_processor::setRMReg8(Dsim_instruction *curr_ins, int n) {
    switch(curr_ins->getRm()) {
      case 0x00:
        EAX->setLL8(n & 0xff);
        break;
      case 0x01:
        ECX->setLL8(n & 0xff);
        break;
      case 0x02:
        EDX->setLL8(n & 0xff);
        break;
      case 0x03:
        EBX->setLL8(n & 0xff);
        break;
      case 0x04:
        EAX->setLU8(n & 0xff);
        break;
      case 0x05:
        ECX->setLU8(n & 0xff);
        break;
      case 0x06:
        EDX->setLU8(n & 0xff);
        break;
      case 0x07:
        EBX->setLU8(n & 0xff);
        break;
      }
    }


// reg stuff


  void Dsim_processor::setReg32(Dsim_instruction *curr_ins, int n) {
    
    Dsim_register* r1 = get_register(curr_ins->getReg());
    r1->setData(n);
    }
  void Dsim_processor::setReg16(Dsim_instruction *curr_ins, int n) {
    
    Dsim_register* r1 = get_register(curr_ins->getReg());
    r1->setLower16(n & 0xffff);
    }
  int Dsim_processor::getReg16(Dsim_instruction *curr_ins) {
    
    Dsim_register* r1 = get_register(curr_ins->getReg());
    return r1->getLower16();
    }
  int Dsim_processor::getReg32(Dsim_instruction *curr_ins) {
    
    Dsim_register* r1 = get_register(curr_ins->getReg());
    return r1->getData();
    }
  void Dsim_processor::setReg8(Dsim_instruction *curr_ins, int n) {
    switch(curr_ins->getReg()) {
      case 0x00:
        EAX->setLL8(n & 0xff);
        break;
      case 0x01:
        ECX->setLL8(n & 0xff);
        break;
      case 0x02:
        EDX->setLL8(n & 0xff);
        break;
      case 0x03:
        EBX->setLL8(n & 0xff);
        break;
      case 0x04:
        EAX->setLU8(n & 0xff);
        break;
      case 0x05:
        ECX->setLU8(n & 0xff);
        break;
      case 0x06:
        EDX->setLU8(n & 0xff);
        break;
      case 0x07:
        EBX->setLU8(n & 0xff);
        break;
      }
    }
  int Dsim_processor::getReg8(Dsim_instruction *curr_ins) {
    switch(curr_ins->getReg()) {
      case 0x00:
        return EAX->getLL8();
        break;
      case 0x01:
        return ECX->getLL8();
        break;
      case 0x02:
        return EDX->getLL8();
        break;
      case 0x03:
        return EBX->getLL8();
        break;
      case 0x04:
        return EAX->getLU8();
        break;
      case 0x05:
        return ECX->getLU8();
        break;
      case 0x06:
        return EDX->getLU8();
        break;
      case 0x07:
        return EBX->getLU8();
        break;
      }
    return ~0;
    }
/**
 * Execute a passed instruction.
 **/
void
Dsim_processor::execute(Dsim_instruction *ins, int lendis)
{
  /* a variable to tell whether there has been a significant change
     in the flow of execution made by an instruction like say jmp */
  bool flow_changed = false;
  int a, b , c, d , e;

  Dsim_register* r1 = NULL;

  bool operandOveride = false;
  bool repeat = false;
  int prefix = 0;

  for (a = 0; a < ins->get_numprefix(); a++) {
    if ((ins->get_prefix(a) == 0xf3) || (ins->get_prefix(a) == 0xf2)) {
      repeat = true;
      prefix = ins->get_prefix(a);
    }
  }
  if (repeat) { // put repeat instructions here
    switch (prefix) {
      case 0xf3:
        switch(ins->get_opcode()) {
          case 0xa5:
            for (a = 0; a < ECX->getData(); a++) {
              b = getDoubleWord(DS->getData() + ESI->getData());
              putDoubleWord((ES->getData() + EDI->getData()), b);
              ESI->setData(ESI->getData() + 4);
              EDI->setData(EDI->getData() + 4);
              }
          break;
        }
      case 0xf2:
        switch(ins->get_opcode()) {
          case 0x00:
          break;
        }
      }
    }

  else switch(ins->get_opcode()) {
    case 0x00:
      a = getRM8(ins);
      b = getReg8(ins);
      setRM8(ins, (a + b));
      break;
    case 0x01:
      a = getRM32(ins);
      b = getReg32(ins);
      setRM32(ins, (a + b));
      break;
    case 0x02:
      a = getReg8(ins);
      b = getRM8(ins);
      setReg8(ins, (a + b));
      break;
    case 0x03:
      a = getReg32(ins);
      b = getRM32(ins);
      setReg32(ins, (a + b));
      break;
    case 0x04:
      a = EAX->getLL8();
      b = ins->getImmediate();
      EAX->setLL8(a + b);
      break;
    case 0x05:
      a = EAX->getData();
      b = ins->getImmediate();
      EAX->setData((a + b));
      break;
    case 0x06:
      d = ES->getData();
      if((ESP->getData() - 1) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        ESP->setData(ESP->getData() - 1);
        core->insertByte(ESP->getData(), d);
        }
      break;
    case 0x07:
      a = core->inspectByte(ESP->getData());
      ESP->setData(ESP->getData() + 1);
      ES->setData(a);
      break;
    case 0x08:
      a = getRM8(ins);
      b = getReg8(ins);
      setRM8(ins, (a | b));
      break;
    case 0x09:
      a = getRM32(ins);
      b = getReg32(ins);
      setRM32(ins, (a | b));
      break;
    case 0x0a:
      a = getReg8(ins);
      b = getRM8(ins);
      setReg8(ins, (a | b));
      break;
    case 0x0b:
      a = getReg32(ins);
      b = getRM32(ins);
      setReg32(ins, (a | b));
      break;
    case 0x0c:
      a = EAX->getLL8();
      b = ins->getImmediate();
      EAX->setLL8(a | b);
      break;
    case 0x0d:
      a = EAX->getData();
      b = ins->getImmediate();
      EAX->setData(a | b);
      break;
    case 0x0f:
      // 0x20 = 0x20 if zero
      // 0x20 = 0x00 if not zero
      // 0x01 = 0x01 if sign is negative
      switch(ins->get_opcode2()) { // 0x20 = zero
        case 0x83: // sign and zero - jump if above or equal
        
      
    
          //getFlagStatus(int);
          if (!getFlagStatus(0x1)) { // ?
            a = ins->getRelative();
            EIP->setData(EIP->getData() + a);
            }
         break;
       case 0x8d:
         if (getFlagStatus(128) == getFlagStatus(2048)) {
           a = ins->getRelative();
           EIP->setData(EIP->getData() + a);
           }
         break;
       //case 0x95: // ??
         //if ((EFLAG->getData() & 0x20) == 0x00)
           //setRM8(ins, 1);
         //else setRM8(ins, 0);
         //break;  
       case 0x84: // equal
         if (getFlagStatus(64)) {
            a = ins->getRelative();
            EIP->setData(EIP->getData() + a);
            }
         break;
       case 0x85: // not equal
         if (getFlagStatus(64)) {
           a = ins->getRelative();
           EIP->setData(EIP->getData() + a);
           }
         break;

       case 0xb7:
         b = getRM16(ins);
         setReg32(ins, b);
         break;
       default:
         print_def();
         break;
         }
      break;
    case 0x10:
      a = getReg8(ins);
      b = getRM8(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setReg32(ins, (a + b + 1));
      else setRM32(ins, (a + b));
      break;
    case 0x11:
      a = getReg32(ins);
      b = getRM32(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setReg32(ins, (a + b + 1));
      else setRM32(ins, (a + b));
      break;
    case 0x12:
      a = getReg8(ins);
      b = getRM8(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setReg8(ins, (a + b + 1));
      else setReg8(ins, (a + b));
      break;
    case 0x13:
      a = getReg32(ins);
      b = getRM32(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setReg32(ins, (a + b + 1));
      else setReg32(ins, (a + b));
      break;
    case 0x14:
      a = EAX->getLL8();
      b = ins->getImmediate();
      if ((EFLAG->getData() & 0x1) == 0x1)
        EAX->setLL8(a + b + 1);
      else EAX->setLL8(a + b);
      break;
    case 0x15:
      a = EAX->getData();
      b = ins->getImmediate();
      if ((EFLAG->getData() & 0x1) == 0x1)
        EAX->setData(a + b + 1);
      else EAX->setData(a + b);
      break;
    case 0x16:
      d = SS->getData();
      if((ESP->getData() - 1) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        core->insertByte(ESP->getData(), d);
        ESP->setData(ESP->getData() - 1);
        }
      break;
    case 0x17:
      a = core->inspectByte(ESP->getData());
      SS->setData(a);
      ESP->setData(ESP->getData() + 1);
      break;
    case 0x18:
      a = getRM8(ins);
      b = getReg8(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setRM8(ins, (a - b + 1));
      else setRM8(ins, (a - b));
      break;
    case 0x19:
      a = getRM32(ins);
      b = getReg32(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setRM32(ins, (a - b + 1));
      else setRM32(ins, (a - b));
      break;
    case 0x1a:
      a = getReg8(ins);
      b = getRM8(ins);
       if ((EFLAG->getData() & 0x1) == 0x1)
        setReg8(ins, (a - b + 1));
      else setReg8(ins, (a - b));
      break;
    case 0x1b:           // need to add borrow - need to check order of sub
      a = getReg32(ins);
      b = getRM32(ins);
      if ((EFLAG->getData() & 0x1) == 0x1)
        setReg32(ins, (a - b + 1));
      else setReg32(ins, (a - b));
      break;
    case 0x1c:
      a = EAX->getLL8();
      b = ins->getImmediate();
      if ((EFLAG->getData() & 0x1) == 0x1)
        EAX->setLL8(a - b + 1);
      else EAX->setLL8(a - b);
      break;
    case 0x1d:
      a = EAX->getData();
      b = ins->getImmediate();
      if ((EFLAG->getData() & 0x1) == 0x1)
        EAX->setData(a - b + 1);
      else EAX->setData(a - b);
    case 0x1e:
      d = DS->getData();
      if((ESP->getData() - 1) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        core->insertByte(ESP->getData(), d);
        ESP->setData(ESP->getData() - 1);
        }
      break;
    case 0x1f:
      a = core->inspectByte(ESP->getData());
      DS->setData(a);
      ESP->setData(ESP->getData() + 1);
      break;
    case 0x20:
      a = getRM8(ins);
      b = getReg8(ins);
      setRM8(ins, (a & b));
      break;
    case 0x21:
      a = getRM32(ins);
      b = getReg32(ins);
      setRM32(ins, (a & b));
      break;
    case 0x22:
      a = getReg8(ins);
      b = getRM8(ins);
      setReg8(ins, (a & b));
      break;
    case 0x23:
      a = getReg32(ins);
      b = getRM32(ins);
      setReg32(ins, (a & b));
      break;
    case 0x24:
      a = EAX->getLL8();
      b = ins->getImmediate();
      EAX->setLL8(a & b);
      break;
    case 0x25:
      for (a = 0;a < ins->get_numprefix(); a++) {
        if (ins->get_prefix(a) == 0x66) operandOveride = true;
        }

      if (operandOveride) {
        a = EAX->getLower16();
        b = ins->getImmediate();
        EAX->setLower16(a & b);
        }
      else {
        a = EAX->getData();
        b = ins->getImmediate();
        EAX->setData(a & b);
        }
      break;
    case 0x28:
      a = getRM8(ins);
      b = getReg8(ins);
      setRM8(ins, (a - b));
      break;
    case 0x29:
      a = getRM32(ins);
      b = getReg32(ins);
      setRM32(ins, (a - b));
      break;
    case 0x2a:
      a = getReg8(ins);
      b = getRM8(ins);
      setReg8(ins, (a - b));
      break;
    case 0x2b:
      a = getReg32(ins);
      b = getRM32(ins);
      setReg32(ins, (a - b));
      break;
    case 0x2c:
      a = EAX->getLL8();
      b = ins->getImmediate();
      EAX->setLL8(a - b);
      break;
    case 0x2d:
      a = EAX->getData();
      b = ins->getImmediate();
      EAX->setData(a - b);
      break;
    case 0x30:
      a = getRM8(ins);
      b = getReg8(ins);
      setRM8(ins, (a ^ b));
      break;
    case 0x31:
      a = getRM32(ins);
      b = getReg32(ins);
      setRM32(ins, (a ^ b));
      break;
    case 0x32:
      a = getReg8(ins);
      b = getRM8(ins);
      setReg8(ins, (a ^ b));
      break;
    case 0x33:
      a = getReg32(ins);
      b = getRM32(ins);
      setReg32(ins, (a ^ b));
      break;
    case 0x34:
      a = EAX->getLL8();
      b = ins->getImmediate();
      EAX->setLL8(a ^ b);
      break;
    case 0x35:
      a = EAX->getData();
      b = ins->getImmediate();
      EAX->setData(a ^ b);
      break;
    case 0x38:
      a = getRM8(ins);
      b = getReg8(ins);
      setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
      break;
    case 0x39:
      a = getRM32(ins);
      b = getReg32(ins);
      setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
      break;
    case 0x3c:
      a = EAX->getLL8();
      b = ins->getImmediate();
      setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
      break;
    case 0x3d:
      a = EAX->getData();
      b = ins->getImmediate();
      setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
      break;
    case 0x40: case 0x41: case 0x42: case 0x43:
    case 0x44: case 0x45: case 0x46: case 0x47:
      r1 = get_register((int)ins->get_opcode() - 0x40);
      r1->setData(r1->getData() + 1);
      break;
    case 0x48: case 0x49: case 0x4a: case 0x4b:	
    case 0x4c: case 0x4d: case 0x4e: case 0x4f:
      r1 = get_register((int)ins->get_opcode() - 0x48);
      r1->setData(r1->getData() - 1);
      break;
    case 0x50 : case 0x51 : case 0x52 : case 0x53 :
    case 0x54 : case 0x55 : case 0x56 : case 0x57 :
      r1 = get_register(ins->get_opcode() - 0x50);
      d = r1->getData();
      if((ESP->getData() - 4) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        ESP->setData(ESP->getData() - 4);
        putDoubleWord(ESP->getData(), d);
        }
      break;
     case 0x58 : case 0x59 : case 0x5A : case 0x5B : // pop
     case 0x5C : case 0x5D : case 0x5E : case 0x5F :
        r1 = get_register(ins->get_opcode() - 0x58);
        a = getDoubleWord(ESP->getData());
        ESP->setData(ESP->getData() + 4);
        r1->setData(a);
	break;

    case 0x68:
      d = ins->getImmediate();
      if((ESP->getData() - 4) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        ESP->setData(ESP->getData() - 4);
        putDoubleWord(ESP->getData(), d);
        }
      break;

    case 0x6a:
      d = ins->getImmediate();
      if((ESP->getData() - 1) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        ESP->setData(ESP->getData() - 2);
        putWord(ESP->getData(), d); // a word is put on, this is correct
        }
      break;
    case 0x72:
      if (!getFlagStatus(1)) {
        a = ins->getRelative();
        EIP->setData(EIP->getData() + a);
        }
      break;
    case 0x73: // sign and zero - jump if above or equal
      if (!getFlagStatus(1)) {
        a = ins->getRelative();
        EIP->setData(EIP->getData() + a);
        }
      break;
    case 0x74:
      if (getFlagStatus(64)) {
        a = ins->getRelative();
        EIP->setData(EIP->getData() + a);
        }
      break;
    case 0x75:
      if (!getFlagStatus(64)) {
        a = ins->getRelative();
        EIP->setData(EIP->getData() + a);
        }
      break;
    case 0x76:
      if (getFlagStatus(64) | getFlagStatus(1)) {
        a = ins->getRelative();
        EIP->setData(EIP->getData() + a);
        }
      break;
    case 0x7e:
      if (getFlagStatus(64) | (getFlagStatus(128) !=  getFlagStatus(2048))) {
        a = ins->getRelative();
        EIP->setData(EIP->getData() + a);
        }
      break;
    case 0x80:
      switch(ins->getReg()) {
        case 0x00:
          a = getRM8(ins);
          b = ins->getImmediate();
          setRM8(ins, a + b);
          break;
        case 0x01:
          a = getRM8(ins);
          b = ins->getImmediate();
          setRM8(ins, a | b);
          break;
        case 0x02:
          a = getRM8(ins);
          b = ins->getImmediate();
          if ((EFLAG->getData() & 0x1) == 0x1)
            setRM8(ins, (a + b + 1));
          else setRM8(ins, (a + b));
          break;
        case 0x03:
          a = getRM8(ins);
          b = ins->getImmediate();
          if ((EFLAG->getData() & 0x1) == 0x1)
            setRM8(ins, (a - b + 1));
          else setRM8(ins, (a - b));
          break;
        case 0x04:
          a = getRM8(ins);
          b = ins->getImmediate();
          setRM8(ins, a & b);
          break;
        case 0x05:
          a = getRM8(ins);
          b = ins->getImmediate();
          setRM8(ins, a - b);
        case 0x06:
          a = getRM8(ins);
          b = ins->getImmediate();
          setRM8(ins, a ^ b);
          break;
        case 0x07:
          a = getRM8(ins);
          b = ins->getImmediate();
          setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
          break;
        }
    case 0x81:
      switch(ins->getReg()) {
        case 0x00:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, a + b);
          break;
        case 0x01:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, a | b);
          break;
        case 0x02:
          a = getRM32(ins);
          b = ins->getImmediate();
           if ((EFLAG->getData() & 0x1) == 0x1)
            setRM32(ins, (a + b + 1));
          else setRM32(ins, (a + b));
          break;
        case 0x03:
          a = getRM32(ins);
          b = ins->getImmediate();
          if ((EFLAG->getData() & 0x1) == 0x1)
            setRM32(ins, (a - b + 1));
          else setRM32(ins, (a - b));
          break;
        case 0x04: // sign extended
          for (a = 0;a < ins->get_numprefix(); a++) {
            if (ins->get_prefix(a) == 0x66) operandOveride = true;
            }
          b = ins->getImmediate();
          if (operandOveride) {
            a = getRM16(ins);
            if (b > 0x7f00)
            b = b + 0xffff0000;
            setRM16(ins, a & b);
            }
          else {
            a = getRM32(ins);
            setRM32(ins, a & b);
            }
          break;
        case 0x05:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, (a - b));
	  break;
        case 0x06:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, a ^ b);
          break;
        case 0x07:
          a = getRM32(ins);
          b = ins->getImmediate();
          setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
          break;
        }
      break;
    case 0x83:
      switch(ins->getReg()) {
        case 0x00:
          a = getRM32(ins);
          b = ins->getImmediate();
          if (b > 0x7f) setRM32(ins, a - ((b ^ 0xff) + 1));
          else setRM32(ins, a + b);
          break;
        case 0x01:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, a | b);
          break;
        case 0x02:
          a = getRM32(ins);
          b = ins->getImmediate();
          if ((EFLAG->getData() & 0x1) == 0x1) {
            if (b > 0x7f) setRM32(ins, a - ((b ^ 0xff) + 2)); // addinng carry
            else setRM32(ins, a + b + 1);
            }
          else {
            if (b > 0x7f) setRM32(ins, a - ((b ^ 0xff) + 1)); // addinng carry
            else setRM32(ins, a + b);
            }
          break;
        case 0x03:
          a = getRM32(ins);
          b = ins->getImmediate();
          if ((EFLAG->getData() & 0x1) == 0x1)
            setRM32(ins, (a - b + 1));
          else setRM32(ins, (a - b));
          break;
        case 0x04: // sign extended
          a = getRM32(ins);
          b = ins->getImmediate();
          if (b > 127)
            b = b + 0xffffff00;
          setRM32(ins, a & b);
          break;
        case 0x05:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, a - b);
          break;
        case 0x06:
          a = getRM32(ins);
          b = ins->getImmediate();
          setRM32(ins, a ^ b);
          break;
        case 0x07:
          a = getRM32(ins); // first source
          b = ins->getImmediate(); // second source, sign extended
          if (b > 127)
            b = b + 0xffffff00;
          setFlags32((long long)a, (long long)b, ((long long)a - (long long)b));
          break;
        }
      break;
    case 0x84:
      a = getRM8(ins);
      b = getReg8(ins);
      setFlags32(b & a);
      break;
    case 0x85:
      a = getRM32(ins);
      b = getReg32(ins);
      setFlags32(b & a);
      break;
    case 0x88:
      a = getReg8(ins);
      setRM8(ins, a);
      break;
    case 0x89:
      for (a = 0;a < ins->get_numprefix(); a++) {
        if (ins->get_prefix(a) == 0x66) operandOveride = true;
        }
      if (operandOveride) {
        a = (getReg32(ins) & 0xffff);
        setRM16(ins, a);
        }
      else {
        a = getReg32(ins);
        setRM32(ins, a);
        }
      break;
    case 0x8a:
      a = getRM8(ins);
      setReg8(ins, a);
      break;
    case 0x8b:
      for (a = 0;a < ins->get_numprefix(); a++) {
        if (ins->get_prefix(a) == 0x66) operandOveride = true;
        }
      if (operandOveride) {
        a = getRM16(ins);
        setReg16(ins, a);
        }
      else {
        a = getRM32(ins);
        setReg32(ins, a);
        }

      break;
    case 0x8d:
      a = getRM32Address(ins);
      setReg32(ins, a);
      break;
    case 0xa1:
      a = ins->getImmediate();
      EAX->setData(getDoubleWord(a));
      break;
    case 0x90: // NOP instruction , do nothing
      break;
    case 0xa3:
      a = EAX->getData();
      b = ins->getImmediate();
      putDoubleWord(b, a);
      break;

    case 0xa4:
      a = getByte(ESI->getData());
      putByte(EDI->getData(), a);
      ESI->setData(ESI->getData() + 1);
      EDI->setData(EDI->getData() + 1);
      break;

    case 0xa5:
      if (ins->get_numprefix() != 0) {
        for (a = 0; a < ins->get_numprefix(); a++) {
          if (ins->get_prefix(a) == 0x66) operandOveride = true;
          }
        }
      if (operandOveride) {
        a = getWord(ESI->getData());
        putWord(EDI->getData(), a);
        ESI->setData(ESI->getData() + 2);
        EDI->setData(EDI->getData() + 2);
        }
      else {
        a = getDoubleWord(ESI->getData());
        putDoubleWord(EDI->getData(), a);
        ESI->setData(ESI->getData() + 4);
        EDI->setData(EDI->getData() + 4);
        }
      break;

    case 0xb0: case 0xb1: case 0xb2: case 0xb3: // imm8	
    case 0xb4: case 0xb5: case 0xb6: case 0xb7: // imm8
      a = ins->getImmediate();
      r1 = get_register((int)ins->get_opcode() - 0xb0);
      r1->setData(a);
      break;

    case 0xb8: case 0xb9: case 0xba: case 0xbb: // imm32
    case 0xbc: case 0xbd: case 0xbe: case 0xbf: // imm32
      a = ins->getImmediate();
      r1 = get_register((int)ins->get_opcode() - 0xb8);
      r1->setData(a);
      break;
      
    case 0xc1:
      switch (ins->getReg()) {
        case 0x00: print_def(); break;
  	  case 0x01: print_def(); break;
  	  case 0x02: print_def(); break;
        case 0x03: print_def(); break;

	  case 0x04: print_def(); break;
	  case 0x05: 
	    a = getRM32(ins);
	    b = ins->getImmediate();
	    a = a / (2 * b);
	    setRM32(ins, a);
	    break;
	  case 0x06: print_def(); break;
	  case 0x07: print_def(); break;
  	  }
      break;
    case 0xc3:
      flow_changed = true;
      a = getDoubleWord(ESP->getData());
      ESP->setData(ESP->getData() + 4);
      EIP->setData(a);
      indentValue = indentValue - 2;
      break;
    case 0xC6:
      a = ins->getImmediate();
      setRM8(ins, a);
      break;
    case 0xc7:
      a = ins->getImmediate();
      setRM32(ins, a);
      break;
    case 0xd9:
      switch(ins->getReg()) {
        case 0x01: print_def(); break;
        case 0x02: print_def(); break;
        case 0x03: print_def(); break;
        case 0x04: print_def(); break;
        case 0x05:
          a = getRM16(ins);
          FPUControl->setData(a);
          break;
        case 0x06: print_def(); break;
        case 0x07:
          a = FPUControl->getData();
          setRM16(ins, a);
          break;
        }
      break;
    case 0xe8: case 0xe9:
      a = ins->getRelative();
    
      d = EIP->getData() + lendis;
      if((ESP->getData() - 4) < 0)
        Dsim_error("stack overflow error, increase stack size ", true);
      else {
        ESP->setData(ESP->getData() - 4);
        putDoubleWord(ESP->getData(), d);
        EIP->setData(EIP->getData() + a);
        indentValue++;
        indentValue++;
        
        }
      break;

    case 0xcd:
      
      EAX->setData(sys->sys(EAX->getData()));
      break;
    case 0xeb:
      a = ins->getRelative();
      EIP->setData(EIP->getData() + a);
      break;

    case 0xfc:
      if ((EFLAG->getData() & 0x200) == 0x200)
        EFLAG->setData(EFLAG->getData() ^ 0x200);
      break;
    case 0xf6: // check;
      switch(ins->getReg()) {
        case 0x00:
          a = getRM8(ins);
          b = ins->getImmediate();
          setFlags32(b & a);
          break;
        case 0x01: print_def(); break;
        case 0x02: print_def(); break;
        case 0x03: print_def(); break;
        case 0x04: print_def(); break;
        case 0x05: print_def(); break;
        case 0x06: print_def(); break;
        case 0x07: print_def(); break;
        }
      break;
    case 0xf7:
      switch(ins->getReg()) {
        case 0x00: print_def(); break;
        case 0x01: print_def(); break;
        case 0x02:
          a = getRM32(ins);
          setRM32(ins, (a ^ 0xffffffff));
          break;
        case 0x03:
          a = getRM32(ins);
          setRM32(ins, (a ^ 0xffffffff) + 1);
          break;
        case 0x04: print_def(); break;
        case 0x05: print_def(); break;
        case 0x06:  // divide instruction fully doesn't work, issues !!
          a = EDX->getData();
          b = EAX->getData();
          c = getRM32(ins);
          d = ((a / c) << 16) + (b / c);
          EAX->setData(d);
          while (e > c) {
            e = a - c;
            }
          b = b + e;
          while (e > c) {
            e = b - c;
            }
          EDX->setData(e);
          break;
        case 0x07: print_def(); break;
        }
      break;

   case 0xff:
     switch(ins->getReg()) {
        case 0x01:
          a = getRM32(ins);
          setRM32(ins, a + 1);
          break;print_def();
        case 0x02:
          d = EIP->getData() + lendis;
          flow_changed = true;
          a = getRM32(ins);
          EIP->setData(a);
          indentValue++;
          indentValue++;
          if((ESP->getData() - 4) < 0)
            Dsim_error("stack overflow error, increase stack size ", true);
          else {
            ESP->setData(ESP->getData() - 4);
            putDoubleWord(ESP->getData(), d);
            }
          break;
        case 0x03: print_def(); break;
        case 0x04: print_def(); break;
        case 0x05: print_def(); break;
        case 0x06: print_def(); break;
        case 0x07: print_def(); break;
        }
      break;

//---------- by default opcode has not been implemented ------------
      default :         
        print_def();  
        break;
     }
   if(!flow_changed) {
     EIP->setData(EIP->getData() + lendis);
     }
   else flow_changed = false;


   operandOveride = false;
   repeat = false;

// for debugging purposes only
/* 
   indent();
   cout << "EAX = " << EAX->getData() << " ECX = " << ECX->getData() << " EFLAG = " << EFLAG->getData() << "\n";
   indent();
   cout << "EDX = " << EDX->getData() << " EBX = " << EBX->getData() << " ESP = " << ESP->getData() << "\n";
   indent();
   cout << "EBP = " << EBP->getData() << " ESI = " << ESI->getData() << " EDI = " << EDI->getData() << "\n";
   int addr = 0xff20;
   indent();
   cout << "Stack\n";
   for (int i = 0; (addr + (i * 4)) <= core->getInitStackValue() ; i++) {
     indent();
     cout << addr + (i * 4) << " " << core->inspectByte(addr + 0 + (i * 4)) << " " << core->inspectByte(addr + 1 + (i * 4)) << " " << core->inspectByte(addr + 2 + (i * 4)) << " " << core->inspectByte(addr + 3 + (i * 4)) << "\n";
     }*/

}

void Dsim_processor::indent() {
  int i = 0;
  for (i = 0; i < indentValue; i++)
    cout << " ";
  }

/**
 * cleanup is the actual function that does the 
 * the destruction . Called by the destructor.
 **/
void
Dsim_processor::cleanup()
{
}

/**
 * prints the default message for the execution
 * saying that the opcode has not been implemented
 **/
void
Dsim_processor::print_def()
{
   indent();
   cout << "Sorry opcode not implemented !! " << endl;
}


/**
 * takes in a index, and returns the entire register
 * associated with the name. Note, to get hold of
 * data or lower register such as ax or al , will
 * need to use the getLower(), setLower etc. methods
 * from the attained register.
 **/ 
Dsim_register*
Dsim_processor::get_register(int regnum)
{
   
   if(regnum == 0) {

     return EAX;}
   if(regnum == 1) {

     return ECX;}
   if(regnum == 2) {

     return EDX;}
   if(regnum == 3) {

     return EBX;}
   if(regnum == 4) {

     return ESP;}
   if(regnum == 5) {

     return EBP;}
   if(regnum == 6) {

     return ESI;}
   if(regnum == 7) {

     return EDI;}
   
   return NULL;
}

/**
 * deep copying . 
 **/
void
Dsim_processor::copy(const Dsim_processor&)
{
   assert(0);
}

/**
 * For things like Dsim_processor a = b;
 * overloaded operator = 
 **/
Dsim_processor&
Dsim_processor::operator=(const Dsim_processor& c)
{
   // making sure that the passed parameter 
   // is not itself.
   if ( this != &c ) {	
      // do a cleanup and copy.	
      cleanup(); copy(c);
   }

   return *this;
}


