/***********************************************************************
                          Dsim_elfer.cc
                          Peter McClymont
                     University of Auckland
************************************************************************/
#include <iostream.h>
#include <stdio.h>
#include <elf.h>
#include <libelf/libelf.h>
#include <fcntl.h>
#include <fstream.h>

#include "Dsim_elfer.h"
#include "GetLongOpt.h"
#include "Dsim_error.h"
#include "Dsim_global.h"
#include "Dsim_types.h"
#include "Dsim_core.h"

extern Dsim_core core;

/**
 * overloaded operator << for use normally with cout.
 **/
ostream&
operator<<(ostream& os, const Dsim_elfer&)
{
   return os;
}


/**
 * main initial constructor.
 **/
Dsim_elfer::Dsim_elfer(const char *filename)
{
   file_name=filename;
   if(validate())
   {
      //cout << "initial input file validation sucessful" << endl;
      start_reading();
   }
}

/**
 * returns the entry point into the program
 * it also sometimes corresponds to the start of the 
 * .text section and the text segment.
 **/
Dsim_address
Dsim_elfer::getEntry()
{
   return ehdr->e_entry;
}

/**
 * returns the entry point into the data segment
 * it corresponds to the start of the .text 
 * section.
 **/
Dsim_address
Dsim_elfer::getTextSegEntry()
{
   return textEntry;
}

/**
 * returns the offset into the file till
 * the text segment
 **/
int
Dsim_elfer::getTextSegOffset()
{
   return textOffset;
}

/**
 * returns the entry point into the data segment
 * it corresponds to the start of the .data 
 * section.
 **/
Dsim_address
Dsim_elfer::getDataSegEntry()
{
  return dataEntry;
}

/**
 * returns the entry point into the bss segment
 * it corresponds to the start of the .bss
 * section.
 **/
Dsim_address
Dsim_elfer::getBssSegEntry()
{
  return bssEntry;
}

/**
 * returns the size of the text segment.
 * size of the text segment is the size of 
 * the .text section + size of all the sections
 * till .data (i.e. the entire text segment).
 **/
long
Dsim_elfer::getTextSegSize()
{
   return textSize; 
}

/**
 * returns the size of the data segment, works
 * the same way as the text segment.
 **/
long
Dsim_elfer::getDataSegSize()
{
   return dataSize; 
}

/**
 * retuns the size of the bss segment. Not 
 * quite sure how necessary it is, might be
 * removed later ?
 **/
long
Dsim_elfer::getBssSegSize()
{
   return bssSize + 20; // not correct, but not critical like the others
}


/**
 * Obtains the basic ELF information and does
 * various validation checks on different initial
 * things that can go wrong. It should be called
 * straight from the constructor.
 **/
bool 
Dsim_elfer::validate()
{
   int fd;
   /* Open the input file */
   if ((fd = open(file_name, O_RDONLY)) == -1)
   {
      Dsim_error err("error opening input file",true);
   }
  	
   /* Obtain the ELF descriptor */
   (void) elf_version(EV_CURRENT);

   if ((elf = elf_begin(fd, ELF_C_READ, NULL)) == NULL)
   {
      Dsim_error err("error obtaining the ELF descriptor in file",true);
   }
   	
   /* Obtain the ELF header */
   if ((ehdr = elf32_getehdr(elf)) == NULL)
   {
      Dsim_error err("error obtaining the ELF header in file ",true);
   }
   	
   /* Obtain the program header */
   if((phdr = elf32_getphdr(elf)) == NULL)
   {
      Dsim_error err("error obtaining the program header in file ",true);
   }
   	
   /*Obtain the .shstrtab data buffer */	
   if((scn = elf_getscn(elf, ehdr->e_shstrndx)) == NULL)
   {
      Dsim_error err("error obtaining the .shstrtab data bufffer ",true);
   }
 
   if((data = elf_getdata(scn, NULL)) == NULL)
   {
      Dsim_error err("error obtaining the .shstrtab data bufffer ",true);
   }
   	
   /* Try to ascertain whether the file is executable or not */
   if(ehdr->e_type!=ET_EXEC)
   {
      Dsim_error("The input file is not an executable file",true);
   }
   	
   /* Try to ascertain whether the file is valid for X86 architecture */
   if(ehdr->e_machine!=EM_386)
   {
      Dsim_error err("The input file is not of x86 architecture",true);
   }

   return true;
}

/**
 * fills the passed character buffer with all
 * the instructions of the text segment.
 **/
void
Dsim_elfer::fill_text(char* store)
{
   ifstream infile;
   infile.open(file_name, ios::in);

   if ( infile.fail() ) {
      cerr << "Cannot open " << file_name << " for loading" << endl;
      exit(-1);
   } 
   infile.seekg(textOffset);
   infile.read(store , getTextSegSize());
   infile.close();
   //cout << "text size = " << getTextSegSize() << "\n";
   //cout << "text offset = " << textOffset << "\n";
}

/**
 * fills the passed character buffer with all
 * the instructions of the data segment.
 **/
void
Dsim_elfer::fill_data(char* store)
{
   ifstream infile;
   infile.open(file_name, ios::in);

   if ( infile.fail() ) {
      cerr << "Cannot open " << file_name << " for loading" << endl;
      exit(-1);
   } 
 
   //cout << "data offset is " << dataOffset << endl;
   //cout << "data size is " << getDataSegSize() << endl;

   infile.seekg(dataOffset);
   infile.read(store , getDataSegSize());
   infile.close();

   /*cout << "----------- Reading Data ----------\n";
   for (int i= 0; i < 100; ++i) {
      cout << "byte " << i << ": " << hex << int(store[i]) << endl;
   }

   cout << "----------- Data Found ----------\n";
   for (int i= 0; i < 100; ++i) {
      cout << "byte " << i << ": " << hex 
           << int(core.inspectByte(0x0805de70 +i)) << endl;
   }*/

}

/**
 * Starting to read a binary file.Does an
 * enumeration through all the sections in the
 * executable file and gathers some basic information
 * like size of segments , thier offests within the file
 * It must be called in the constructor after initial 
 * validation has been compeleted successfully.
 **/
void
Dsim_elfer::start_reading()
{
   bool textSegmentReached = false,
	dataSegmentReached = false,
	bssSegmentReached  = false;

   //printf("Entry point : 0x%x \n\n",ehdr->e_entry);
   // all the printing will go away later.
   /* printing the program header information */

   /*printf("Program header info:\n");
   printf("p_type\t\t Ox%x\n", phdr->p_type);
   printf("p_offset\t Ox%x\n", phdr->p_offset);
   printf("p_vaddr\t\t Ox%x\n", phdr->p_vaddr);
   printf("p_paddr\t\t Ox%x\n", phdr->p_paddr);
   printf("p_filesz\t Ox%x\n", phdr->p_filesz);
   printf("p_memsz\t\t Ox%x\n\n", phdr->p_memsz);
   */

   /* printing information on sections */
   // printf("Program sections info:\n");
   /* the heading line */
   //printf("order   sh_name    sh_size        sh_off        sh_addr\n");
   int textSize1 = 0;
   int textSize2 = 0;
   int dataSize1 = 0;
   int dataSize2 = 0;
   scn = 0;
   for (int cnt = 1; (scn = elf_nextscn(elf, scn)); cnt++)
   {
      if ((shdr = elf32_getshdr(scn)) == NULL){
	 printf("error\n");
	 return;
      }

      if(strcmp(((char *)data->d_buf + shdr->sh_name),".init")==0){
	 textSegmentReached=true;
         textEntry = shdr->sh_addr;
	 textOffset = shdr->sh_offset;
      }
      else if(strcmp(((char *)data->d_buf + shdr->sh_name),".data")==0){
         
         textSize += (shdr->sh_offset - textSize2);
	 
	 
	 
	 textSegmentReached = false;	
	 dataSegmentReached=true;
	 dataEntry = shdr->sh_addr;
         dataOffset = shdr->sh_offset;
         
         
      }
      else if(strcmp(((char *)data->d_buf + shdr->sh_name),".bss")==0){
         dataSize += (shdr->sh_offset - dataSize2);
	 dataSegmentReached = false;
	 bssSegmentReached=true;
	 bssEntry = shdr->sh_addr;
      }

      Elf_Data *text_data=0;
      Elf_Data *data_data=0;
      Elf_Data *bss_data=0;
      // ok start reading now once text section is reached...
      if(textSegmentReached){
	 // now try to manipulate data from section.
	 if((text_data = elf_getdata(scn,text_data))==NULL ){
	    Dsim_error err("error in trying to get data for text segment");
	 }
	 /*printf("ok printing about the type of data now ::\n");
	 printf("d_type is 0x%x \n",text_data->d_type);
	 printf("d_size is 0x%x \n",text_data->d_size);
	 printf("d_off is  0x%x \n",text_data->d_off);
	 */ 		
	 //textSectionReached=false;
	 /*printf("adding %x to %x gives %x \n",textSize,text_data->d_size,
					      textSize+text_data->d_size);
	 */
	 
         
	 textSize1 = shdr->sh_offset;
	 
	 
	 if (textSize2 != 0)
	   textSize+= textSize1 - textSize2; //text_data->d_size;
	 textSize2 = textSize1;
         

	
	
	 
	 
	 
	
         
	
	
	
      }
      else if(dataSegmentReached){
	 if((data_data = elf_getdata(scn,data_data))==NULL ){
	    Dsim_error err("error in trying to get data for data segment");
	 }
         dataSize1 = shdr->sh_offset;



         if (dataSize2 != 0)
           dataSize += dataSize1 - dataSize2;
         dataSize2 = dataSize1;
	 
  //       cout << "Data Segment/n";
    //     for(int i=0;i<data_data->d_size;i++){
      //     char* point = (char *)(text_data->d_buf);
        //   printf("0x%x\n",*(point+i));
        // }
      }
      else if(bssSegmentReached){
	 if((bss_data = elf_getdata(scn,bss_data))==NULL ){
	    Dsim_error err("error in trying to get data for bss segment");
	 }
	 bssSize += bss_data->d_size;
      }
      /*	
      (void) printf("[%d]    %s", cnt,
		  (char *)data->d_buf + shdr->sh_name);
      */
	
      /* printing some spaces for tabular form */
      /*for(int i = 0;i < 15-strlen((char *)data->d_buf + shdr->sh_name); i++)
	 printf(" ");
       */
      /* printing the sh_size and sh_addr */
      /*printf("Ox%x           Ox%x        0x%x",
	    shdr->sh_size,shdr->sh_offset,shdr->sh_addr);
      printf("\n");
      */
   }
}

/**
 * cleanup is the actual function that does the 
 * the destruction . Called by the destructor.
 **/
void
Dsim_elfer::cleanup()
{
}

/**
 * deep copying . 
 **/
void
Dsim_elfer::copy(const Dsim_elfer&)
{
   assert(0);
}

/**
 * For things like Dsim_elfer a = b;
 * overloaded operator = 
 **/
Dsim_elfer&
Dsim_elfer::operator=(const Dsim_elfer& c)
{
   // making sure that the passed parameter 
   // is not itself.
   if ( this != &c ) {	
      // do a cleanup and copy.	
      cleanup(); copy(c);
   }

   return *this;
}

