/***********************************************************************
                          Dsim_core.cc
                          Peter McClymont
                     University of Auckland
************************************************************************/
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>
#include <assert.h>


#include "Dsim_core.h"
#include "GetLongOpt.h"
#include "Dsim_elfer.h"
#include "Dsim_global.h"
#include "Dsim_types.h"
#include "Dsim_misclib.h"
#include "Dsim_error.h"


/**
 * The names of the segments to look out for.
 *
 * SYS_SEG is the system segment which 
 * holds the stack.
 * TEXT_SEG is the text segment which holds 
 * the program code.
 * DATA_SEG is the data segment which holds 
 * all the initialised data.
 * BSS_SEG is the bss segment which is there 
 * for dynamically created data.
 **/

const char *Dsim_core::segmentName[] = {
   "SYS_SEG","TEXT_SEG", "DATA_SEG", "BSS_SEG"
};  



/**
 * overloaded operator << for use normally with cout.
 **/
ostream&
operator<<(ostream& os, const Dsim_core&)
{
   return os;
}


/**
 * main initial constructor.
 * empty in this case.
 **/
Dsim_core::Dsim_core()
{
}



/**
 * the init method , initializes the core.
 * It builds up an elfer object which in turn reads
 * the file , while the init method instantiates
 * things like the structures to hold the instructions.
 **/
void
Dsim_core::init()
{
   cout << hex << "\n";
   const char* fileName = option.getarg(0);
   
   elfer = new Dsim_elfer(fileName);

   const Dsim_address coreSize =
      (atol(option.retrieve("coreSize")) >> 18) << 18;
       // aligned to 256k byte boundary
   const int stackSize =
      (atoi(option.retrieve("stackSize")) >> 16) << 16;
      // aligned to 64k byte boundary
   
   
   

   if ( coreSize == 0 ) {
      Dsim_error err("Core size must be greater than 256 kbytes",true); 
   }
   if ( stackSize == 0 ) {
      Dsim_error err("Stack size must be greater than 64 kbytes",true);
   }
   
   segmentInfo = new SegmentInfo[numSegments];
   
   bytesPerWord = sizeof(Dsim_data)/sizeof(char);
   

   log2bytesPerWord = int(ilog2(bytesPerWord));

   

   core_size = coreSize >> log2bytesPerWord ; // size in words
   
   core_buffer = new Dsim_data[core_size / 10];
   // buffer stores the core '/ 10' was added because of the memory constraints    //of my home computer, take out.

   

   if (core_buffer == 0 ) {
      Dsim_error err("Panic: core cannot allocate memory",true);
   }      
 
   // LAYOUT OF THE SEGMENTS :
   //
   // ------------------------------  <-- 0x0
   // |      stack         (sys)   |
   // |............................|  <-- stack pointer
   // ------------------------------
   // |........................... |  <-- instruction
   // |     text segment           |      pointer
   // ------------------------------
   // |    data segment            |
   // ------------------------------
   // |      bss segment           |
   // ------------------------------   

   segmentInfo[SYS_SEG].startReal = 0;
   //cout <<" 0x" <<  hex << segmentInfo[SYS_SEG].startReal << endl;
   segmentInfo[SYS_SEG].startMapped = 0;
   segmentInfo[SYS_SEG].sz = stackSize ;
   segmentInfo[SYS_SEG].endMapped = segmentInfo[SYS_SEG].sz - 1;
   segmentInfo[SYS_SEG].endReal = segmentInfo[SYS_SEG].startReal
       + segmentInfo[SYS_SEG].sz - 1;
   segmentInfo[SYS_SEG].disp = segmentInfo[SYS_SEG].startMapped
       - segmentInfo[SYS_SEG].startReal;

   // filling the stack pointer and instruction pointer
   spValue = virtual_address(((stackSize - 1) >> log2bytesPerWord)
                               << log2bytesPerWord);
   //ipValue = virtual_address(((stackSize + 2) >> log2bytesPerWord)
   //                            << log2bytesPerWord);

   //cout << hex << "stack pointer  = " << spValue << endl;
   //cout << hex << "instruction pointer = " << get_entry() << endl;

}


/**
 * stackArgvEtc takes care of all the passed parameter
 * arguments and the system environment variables.
 * (Currently being ignored to move on with other more
 * urgent things).
 **/
void
Dsim_core::stackArgvEtc()
{
}


/**
 * The load method loads up the instructions in the memory
 * structures created in the init method. It is meant to be
 * called immediately after the init method.
 **/
void
Dsim_core::load()
{

   segmentInfo[TEXT_SEG].startReal = elfer->getTextSegEntry();
   segmentInfo[TEXT_SEG].sz = elfer->getTextSegSize();
   segmentInfo[TEXT_SEG].startMapped =
                               segmentInfo[SYS_SEG].endMapped + 1;
   segmentInfo[TEXT_SEG].endMapped = segmentInfo[TEXT_SEG].sz - 1
                               + segmentInfo[TEXT_SEG].startMapped;

   segmentInfo[DATA_SEG].startReal = elfer->getDataSegEntry();
   segmentInfo[DATA_SEG].sz = elfer->getDataSegSize();
   segmentInfo[DATA_SEG].startMapped =
                               segmentInfo[TEXT_SEG].endMapped + 1;
   segmentInfo[DATA_SEG].endMapped = segmentInfo[DATA_SEG].sz - 1
                               + segmentInfo[DATA_SEG].startMapped;

   segmentInfo[BSS_SEG].startReal = elfer->getBssSegEntry();
   segmentInfo[BSS_SEG].sz = elfer->getBssSegSize();
   segmentInfo[BSS_SEG].startMapped =
                               segmentInfo[DATA_SEG].endMapped + 1;
   segmentInfo[BSS_SEG].endMapped = segmentInfo[BSS_SEG].sz - 1
                               + segmentInfo[BSS_SEG].startMapped; 
   
   // filling up the endReal addresses for each of the segments
   for (int s = 0; s < numSegments; ++s ) {
      segmentInfo[s].endReal =
	 segmentInfo[s].startReal + segmentInfo[s].sz - 1;
      segmentInfo[s].disp =
	 segmentInfo[s].startMapped - segmentInfo[s].startReal;

      //cout << "Segment[" << s << endl;
      //cout << "\tstartReal: " << segmentInfo[s].startReal << endl;
      //cout << "\tstartMapped: " << segmentInfo[s].startMapped << endl;
      //cout << "\tendReal: " << segmentInfo[s].endReal << endl;
      //cout << "\tendMapped: " << segmentInfo[s].endMapped << endl;
      //cout << "\tsz: " << segmentInfo[s].sz << endl;
      //cout << "\tdisp: " << segmentInfo[s].disp << endl;
   } 

   // setting the entry point in the core 
   entryPoint = elfer->getEntry();
 
   // bp is the exact point in the core_buffer where 
   // text segments instructions go in.
   char *bp = ((char *)core_buffer) + 
                 (segmentInfo[TEXT_SEG].startMapped); 
 
   // filling up text area with instructions from the file
   elfer->fill_text(bp);    
 
   bp = ((char *)core_buffer) + (segmentInfo[DATA_SEG].startMapped); 
   
   // filling up the data area with instruction from the file
   elfer->fill_data(bp);

   /*for( int i =0 ;i< segmentInfo[DATA_SEG].sz ; i++)
      printf ("0x%x\n",*(bp+i)) ;*/
}

// given a virtual address returns a byte of data in the address

Dsim_data
Dsim_core::inspectByte(Dsim_address addrs) {
  Dsim_address addr = physical_wordaddress(addrs);
  int offset = getOffset(addrs);
  int value = inspect(addr);
  if (offset == 0) {
    return value & 255;
    }
  if (offset == 1) {
    return value >> 8 & 255;
    }
  if (offset == 2) {
    return value >> 16 & 255;
    }
  if (offset == 3) {
    return value >> 24 & 255;
    }
  return ~0;
  }

void
Dsim_core::insertByte(Dsim_address addrs, int byte) {
  Dsim_address addr = physical_wordaddress(addrs);
  int offset = getOffset(addrs);
  int value = 0, value2 = 0;

  if (offset == 3) {
    value = core_buffer[addr] & (0x1000000 - 1);
    core_buffer[addr] = (byte << 24) + value;
    }
  if (offset == 2) {
    value = core_buffer[addr] & (0x10000 - 1);
    value2 = (core_buffer[addr] >> 24) << 24;
    core_buffer[addr] = (byte << 16) + value + value2;
    }
  if (offset == 1) {
    value = core_buffer[addr] & (0x100 - 1);
    value2 = (core_buffer[addr] >> 16) << 16;
    core_buffer[addr] = (byte << 8) + value + value2;
    }
  if (offset == 0) {
    value = core_buffer[addr] >> 8;
    value = value << 8;
    core_buffer[addr] = byte + value;
    }
  }


/**
 * inspect is a method that can be used to inspect something within
 * the core , i.e. given a passed address , it will return the data
 * within the address (in the core). The passed in address should
 * NOT be the virtual address as taken from the excutable , but
 * MUST be the physical address as supplied from the durvasa program.
 * A virtual address MUST be converted to physical address by calling
 * the physical_address() method.
 *
 * Every physical address MUST be a word address and NOT a byte
 * address , this can be acheived by doing a right shift of two
 * bytes inspect(address >> log2bytesPerWord)
 **/
Dsim_data
Dsim_core::inspect(Dsim_address addrs) 
{
   if ( addrs >= Dsim_address(core_size) ) {
      cout << "Invalid memory address: " << hex << addrs << endl;
      exit(-1);
      return 0;
   }
   return core_buffer[addrs];
}


/**
 * replace is used to replace data at any address within the 
 * core with any data (which is also passed in as a parameter).
 * It takes in a physical word address.
 **/
void
Dsim_core::replace(Dsim_address addrs, Dsim_data data)
{
   if ( addrs >= Dsim_address(core_size) ) {
      cerr << "Invalid memory address: " << hex << addrs << endl;
      exit(-1);
      return;
   }
   if ( readOnly(addrs) ) {
      cerr << "Warning: modifying read-only data at address " << hex
          << (addrs << log2bytesPerWord) - segmentInfo[TEXT_SEG].disp
          << endl;
   }
   core_buffer[addrs] = data;
}



/**
 * readOnly tells whether the passed in address is
 * a read only address or not. Currently the criterion
 * is whether the address is within the text segment or
 * not.
 **/
int
Dsim_core::readOnly(Dsim_address addrs) 
{
   Dsim_address byteAddrs = addrs << log2bytesPerWord;
   int rstatus = segmentInfo[TEXT_SEG].startReal <= byteAddrs
                 && segmentInfo[TEXT_SEG].endReal >= byteAddrs;
   return rstatus;
}


/**
 * virtual_address is meant to give a mapping between 
 * the physical address and the virtual address. It
 * takes in the physical byte address and returns the
 * virtual address mapping to it.
 **/
Dsim_address
Dsim_core::virtual_address( Dsim_address pa )
{
   Dsim_address va;

   va = pa;
   for (int m = 0; m < numSegments; ++m ) {
      if (segmentInfo[m].startMapped <= pa &&
         segmentInfo[m].endMapped   >= pa) {
        va = pa - segmentInfo[m].disp;
        return va;
      }
   }
   cerr << "\aCore::virtualAddress: physical address " << hex << pa
       << " is out of range" << endl;
   return ~0;
}



/**
 * physical_address is meant to give a mapping between
 * the virtual address and the physical address. It returns
 * the physical address of the passed virtual address as a
 * byte address.
 **/
Dsim_address
Dsim_core::physical_address( Dsim_address va )
{
   Dsim_address pa;

   pa = va;
   for (int m = 0; m < numSegments; ++m ) {
      if (segmentInfo[m].startReal <= va &&
         segmentInfo[m].endReal   >= va) {
         pa = va + segmentInfo[m].disp; 
         return pa;
      }
   }
   cout << "Core::physicalAddress: virtual address " << hex << va
       << " is out of range" << endl;
   exit(-1);
   return ~0;
}

/**
 * convenience method to get physical word address
 * rather than physical byte address of a given 
 * virtual address.
 **/
Dsim_address
Dsim_core::physical_wordaddress( Dsim_address va )
{
   Dsim_address pa = physical_address(va);
   return pa >> log2bytesPerWord;
}

int
Dsim_core::getOffset(Dsim_address va) {
  return va & 3;
  }

/**
 * returns a character pointer to a passsed virtual address
 * must make sure that the passed in address is a virtual
 * address and NOT a physical address.
 **/
char *
Dsim_core::address2charPtr(Dsim_address va) 
{
   Dsim_address pa;

   if ( (pa = physical_address(va)) == ~0 ) return 0;

   pa = pa >> log2bytesPerWord;       // get word address
   char *ptr = (char *)&core_buffer[pa];

  // ptr += (va & 7);    // add offset

   return ptr;
}

/**
 * returns the entry point into the core , 
 * this is the point from which the simulator
 * must start reading the instructions from.
 **/
Dsim_address
Dsim_core::get_entry()
{
   return entryPoint;
}

/**
 * gets the text segment offset
 **/
int
Dsim_core::get_text_offset()
{
   return elfer->getTextSegOffset();
}

/**
 * returns the size of the text segment
 * in number of bytes.
 **/
int
Dsim_core::get_textseg_size()
{
   return segmentInfo[TEXT_SEG].sz;
}

/**
 * returns the size of the data segment
 * in number of bytes.
 **/
int
Dsim_core::get_dataseg_size()
{
   return segmentInfo[DATA_SEG].sz;
}

/**
 * returns the size of the bss segment
 * in number of bytes.
 **/
int
Dsim_core::get_bssseg_size()
{
   return segmentInfo[BSS_SEG].sz;
}

/**
 * cleanup is the actual function that does the 
 * the destruction . Called by the destructor.
 **/
void
Dsim_core::cleanup()
{
   delete elfer;
   delete [] core_buffer;
}



/**
 * deep copying . 
 **/
void
Dsim_core::copy(const Dsim_core&)
{
   assert(0);
}



/**
 * For things like Dsim_core a = b;
 * overloaded operator = 
 **/
Dsim_core&
Dsim_core::operator=(const Dsim_core& c)
{
   // making sure that the passed parameter 
   // is not itself.
   if ( this != &c ) {	
      // do a cleanup and copy.	
      cleanup(); copy(c);
   }
   return *this;
}

/**
 * returns the initial stack pointer
 * value.
 **/
Dsim_address
Dsim_core::getInitStackValue()
{
   return spValue;
}

/**
 * returns the initial program
 * counter value.
 **/
Dsim_address
Dsim_core::getInitProgValue()
{
   return ipValue;
}
