function [Matbeta, vec_err_n, vec_df, ITC, vec_combr] = map( y, X, mub, nu )

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Input:
% y         = response vector
% X         = full dictionary
% mub       = max. no. of iterations
% nu        = controls how greedy the algorithm is
%Output:
% Matbeta   = p X mub (estimated coefficients)
% vec_err_n = mub X 1 (RSS/n)
% vec_df    = mub X 1 (df)
% ITC       = has mub cols
% Significance of the rows:
% SC1; SC2; ESC1; ESC2; gMDL1; gMDL2; EgMDL1; EgMDL2; BIC; EBIC; AICC
% vec_combr  = mub X 1 (log of binomial coefficient)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[n,p] = size(X);

Matbeta = zeros(p,mub);
vec_err_n = Inf*ones(mub,1);
vec_df = Inf*ones(mub,1);
vec_combr = zeros(mub,1);
ITC = Inf*ones(11,mub);

%Transform data
y = y-mean(y);
for jj=1:p,
    X(:,jj) = X(:,jj) - mean ( X(:,jj) );
    X(:,jj) = sqrt(n)*( X(:,jj)/norm(X(:,jj)) );
end

c = (X'*y)/n;
D = (X'*X)/n;
noy = norm(y)^2/n;
beta = zeros(p,1);
alpha = zeros(p,1);
A = eye(n,n);
s = 0;
df = 0;

for j=1:mub,
    
    if (n==100) && (s>n-1),
        return;
    end
    if (n==20) && (s>5*n-1),
        return;
    end
    if df>n-3,
        return;
    end
    a = c-D*beta;
    [~,sj] = max(abs(a));
    beta(sj) = beta(sj) + nu*a(sj);
    
    est_n = beta'*D*beta;
    err_n = noy + est_n - 2*beta'*c;
    
    if abs(err_n - norm(y-X*beta)^2/n)>10^(-6),
    
    fprintf('%14.10f %14.10f\n', err_n, norm(y-X*beta)^2/n);
    
    end
   
    
    if err_n<=0,
        return;
    end
    Matbeta(:,j) = beta; %save
    vec_err_n(j) = err_n; %save
    if alpha(sj)==0,
        alpha(sj)=1;
        s = sum(alpha);
        if s==1,
            combr = log(p);
        else
            combr = combr + log( (p-s+1)/s );
        end 
    end
    vec_combr(j) = combr;
    P = X(:,sj)*X(:,sj)'/n;
    A = (eye(n,n) - nu*P)*A;  
    df = n-trace(A);
    vec_df(j) = df; %save
    vec_itc = comp_itc(n, noy, est_n, err_n, df, nu, combr);
    ITC(:,j) = vec_itc; %save
end

end %function

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function vec_itc = comp_itc(n, noy, est_n, err_n, df, nu, comb)

%SC
term3 = log(df) -(n-1)*log(n-df);
SC1 = n*log(err_n) +df*log( (est_n/df)/(err_n/(n-df)) ) +term3;
diff_n = noy-err_n;
SC2 = n*log(err_n) +df*log( (diff_n/df)/(err_n/(n-df)) ) +term3;

%ESC
zeta = nu;
ESC1 = SC1 +2*zeta*comb;
ESC2 = SC2 +2*zeta*comb;

%gMDL
gMDL1 = (n/2)*log(noy) + (1/2)*log(n);
gMDL2 = gMDL1;
if (est_n/noy) >= (df/n),
    gMDL1 = (n/2)*log(err_n*n/(n-df)) +(df/2)*log( (est_n/df)/(err_n/(n-df)) ) +log(n);
end
if (diff_n/noy) >= (df/n),
    gMDL2 = (n/2)*log(err_n*n/(n-df)) +(df/2)*log( (diff_n/df)/(err_n/(n-df)) ) +log(n);
end

%EgMDL
EgMDL1 = gMDL1 +zeta*comb;
EgMDL2 = gMDL2 +zeta*comb;

%BIC
BIC = n*log(err_n) +df*log(n);

%EBIC
EBIC = BIC +2*zeta*comb;

%AICC
AICC = log(err_n) +( 1+df/n )/( 1-(df+2)/n );

vec_itc = [SC1; SC2; ESC1; ESC2; gMDL1; gMDL2; EgMDL1; EgMDL2; BIC; EBIC; AICC];

end %function

